% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/engines.R
\name{msprime}
\alias{msprime}
\title{Run a slendr model in msprime}
\usage{
msprime(
  model,
  sequence_length,
  recombination_rate,
  samples = NULL,
  output = NULL,
  random_seed = NULL,
  load = TRUE,
  verbose = FALSE,
  debug = FALSE,
  run = TRUE
)
}
\arguments{
\item{model}{Model object created by the \code{compile} function}

\item{sequence_length}{Total length of the simulated sequence (in base-pairs)}

\item{recombination_rate}{Recombination rate of the simulated sequence (in
recombinations per basepair per generation)}

\item{samples}{A data frame of times at which a given number of individuals
should be remembered in the tree-sequence (see \code{schedule_sampling} for a
function that can generate the sampling schedule in the correct format). If
missing, only individuals present at the end of the simulation will be
recorded in the tree-sequence output file.}

\item{output}{Path to the output tree-sequence file. If \code{NULL} (the default),
tree sequence will be saved to a temporary file.}

\item{random_seed}{Random seed (if missing, SLiM's own seed will be used)}

\item{load}{Should the final tree sequence be immediately loaded and returned?
Default is \code{TRUE}. The alternative (\code{FALSE}) is useful when a tree-sequence
file is written to a custom location to be loaded at a later point.}

\item{verbose}{Write the output log to the console (default \code{FALSE})?}

\item{debug}{Write msprime's debug log to the console (default \code{FALSE})?}

\item{run}{Should the msprime engine be run? If \code{FALSE}, the command line msprime
command will be printed (and returned invisibly as a character vector) but not executed.}
}
\value{
A tree-sequence object loaded via Python-R reticulate interface function \code{ts_load}
(internally represented by the Python object \code{tskit.trees.TreeSequence}). Optionally,
depending on the value of the arguments \code{load =} or \code{run =}, nothing or a character
vector, respectively.
}
\description{
This function will execute a built-in msprime script and run a compiled
slendr demographic model.
}
\examples{
\dontshow{check_dependencies(python = TRUE) # make sure dependencies are present
}
init_env()

# load an example model
model <- read_model(path = system.file("extdata/models/introgression", package = "slendr"))

# afr and eur objects would normally be created before slendr model compilation,
# but here we take them out of the model object already compiled for this
# example (in a standard slendr simulation pipeline, this wouldn't be necessary)
afr <- model$populations[["AFR"]]
eur <- model$populations[["EUR"]]
chimp <- model$populations[["CH"]]

# schedule the sampling of a couple of ancient and present-day individuals
# given model at 20 ky, 10 ky, 5ky ago and at present-day (time 0)
modern_samples <- schedule_sampling(model, times = 0, list(afr, 10), list(eur, 100), list(chimp, 1))
ancient_samples <- schedule_sampling(model, times = c(40000, 30000, 20000, 10000), list(eur, 1))

# sampling schedules are just data frames and can be merged easily
samples <- rbind(modern_samples, ancient_samples)

# run a simulation using the msprime back end from a compiled slendr model object
ts <- msprime(model, sequence_length = 1e5, recombination_rate = 0, samples = samples)

# automatic loading of a simulated output can be prevented by `load = FALSE`, which can be
# useful when a custom path to a tree-sequence output is given for later downstream analyses
output_file <- tempfile(fileext = ".trees")
msprime(model, sequence_length = 1e5, recombination_rate = 0, samples = samples,
        output = output_file, load = FALSE, random_seed = 42)
# ... at a later stage:
ts <- ts_load(output_file, model)

summary(ts)
}
