PGOcc <- function(occ.formula, det.formula, data, inits, priors, 
		  n.samples, n.omp.threads = 1, verbose = TRUE,
		  n.report = 100, n.burn = round(.10 * n.samples), n.thin = 1, 
		  k.fold, k.fold.threads = 1, k.fold.seed = 100, ...){

    ptm <- proc.time()

    # Make it look nice
    if (verbose) {
      cat("----------------------------------------\n");
      cat("\tPreparing the data\n");
      cat("----------------------------------------\n");
    }

    # Functions ---------------------------------------------------------------
    logit <- function(theta, a = 0, b = 1) {log((theta-a)/(b-theta))}
    logit.inv <- function(z, a = 0, b = 1) {b-(b-a)/(1+exp(z))}
    rigamma <- function(n, a, b){
      1/rgamma(n = n, shape = a, rate = b)
    }

    # Check for unused arguments ------------------------------------------
    formal.args <- names(formals(sys.function(sys.parent())))
    elip.args <- names(list(...))
    for(i in elip.args){
        if(! i %in% formal.args)
            warning("'",i, "' is not an argument")
    }
    # Call ----------------------------------------------------------------
    # Returns a call in which all of the specified arguments are
    # specified by their full names.
    cl <- match.call()

    # Some initial checks -------------------------------------------------
    if (missing(data)) {
      stop("error: data must be specified")
    }
    if (!is.list(data)) {
      stop("error: data must be a list")
    }
    names(data) <- tolower(names(data))
    if (missing(occ.formula)) {
      stop("error: occ.formula must be specified")
    }
    if (missing(det.formula)) {
      stop("error: det.formula must be specified")
    }
    if (!'y' %in% names(data)) {
      stop("error: detection-nondetection data y must be specified in data")
    }
    y <- as.matrix(data$y)
    if (!'occ.covs' %in% names(data)) {
      if (occ.formula == ~ 1) {
        if (verbose) {
          message("occupancy covariates (occ.covs) not specified in data.\nAssuming intercept only occupancy model.\n")
        }
        data$occ.covs <- matrix(1, dim(y)[1], 1)
      } else {
        stop("error: occ.covs must be specified in data for an occupancy model with covariates")
      }
    }
    if (!is.matrix(data$occ.covs) & !is.data.frame(data$occ.covs)) {
      stop("error: occ.covs must be a matrix or data frame")
    }
    if (!'det.covs' %in% names(data)) {
      if (det.formula == ~ 1) {
        if (verbose) {
          message("detection covariates (det.covs) not specified in data.\nAssuming interept only detection model.\n")
	}
        data$det.covs <- list(int = rep(1, dim(y)[1]))
      } else {
        stop("error: det.covs must be specified in data for a detection model with covariates")
      }
    }
    if (!is.list(data$det.covs)) {
      stop("error: det.covs must be a list of matrices, data frames, and/or vectors")
    }
    if (missing(n.samples)) {
      stop("error: n.samples must be specified")
    }
    if (n.burn > n.samples) {
      stop("error: n.burn must be less than n.samples")
    }
    if (n.thin > n.samples) {
      stop("error: n.thin must be less than n.samples")
    }
    if (!missing(k.fold)) {
      if (!is.numeric(k.fold) | length(k.fold) != 1 | k.fold < 2) {
        stop("error: k.fold must be a single integer value >= 2")  
      }
    }

    # First subset detection covariates to only use those that are included in the analysis. 
    data$det.covs <- data$det.covs[names(data$det.covs) %in% all.vars(det.formula)]
    # Null model support
    if (length(data$det.covs) == 0) {
      data$det.covs <- list(int = rep(1, dim(y)[1]))
    }
    # Make both covariates a data frame. Unlist is necessary for when factors
    # are supplied. 
    data$det.covs <- data.frame(lapply(data$det.covs, function(a) unlist(c(a))))
    binom <- FALSE
    # Check if all detection covariates are at site level, and simplify the data
    # if necessary
    y.big <- y
    if (nrow(data$det.covs) == nrow(y)) {
     # Convert data to binomial form
     y <- apply(y, 1, sum, na.rm = TRUE) 
     binom <- TRUE
    }
    data$occ.covs <- as.data.frame(data$occ.covs)

    # Checking missing values ---------------------------------------------
    y.na.test <- apply(y.big, 1, function(a) sum(!is.na(a)))
    if (sum(y.na.test == 0) > 0) {
      stop("error: some sites in y have all missing detection histories. Remove these sites from all objects in the 'data' argument, then use 'predict' to obtain predictions at these locations if desired.")
    }

    # Formula -------------------------------------------------------------
    # Occupancy -----------------------
    if (class(occ.formula) == 'formula') {
      tmp <- parseFormula(occ.formula, data$occ.covs)
      X <- as.matrix(tmp[[1]])
      X.re <- as.matrix(tmp[[4]])
      x.re.names <- colnames(X.re)
      x.names <- tmp[[2]]
    } else {
      stop("error: occ.formula is misspecified")
    }

    # Detection -----------------------
    if (class(det.formula) == 'formula') {
      tmp <- parseFormula(det.formula, data$det.covs)
      X.p <- as.matrix(tmp[[1]])
      X.p.re <- as.matrix(tmp[[4]])
      x.p.re.names <- colnames(X.p.re)
      x.p.names <- tmp[[2]]
    } else {
      stop("error: det.formula is misspecified")
    }

    # Get basic info from inputs ------------------------------------------
    # Number of sites
    J <- nrow(y.big)
    # Number of occupancy parameters
    p.occ <- ncol(X)
    # Number of occupancy random effect parameters
    p.occ.re <- ncol(X.re)
    # Number of detection parameters
    p.det <- ncol(X.p)
    # Number of detection random effect parameters
    p.det.re <- ncol(X.p.re)
    # Number of latent occupancy random effect values
    n.occ.re <- length(unlist(apply(X.re, 2, unique)))
    n.occ.re.long <- apply(X.re, 2, function(a) length(unique(a)))
    # Number of latent detection random effect values
    n.det.re <- length(unlist(apply(X.p.re, 2, unique)))
    n.det.re.long <- apply(X.p.re, 2, function(a) length(unique(a)))
    # Number of replicates at each site
    n.rep <- apply(y.big, 1, function(a) sum(!is.na(a)))
    # Max number of repeat visits
    K.max <- max(n.rep)
    # Because I like K better than n.rep
    K <- n.rep

    # Get indices to map z to y -------------------------------------------
    if (!binom) {
      z.long.indx <- rep(1:J, K.max)
      z.long.indx <- z.long.indx[!is.na(c(y.big))]
      # Subtract 1 for indices in C
      z.long.indx <- z.long.indx - 1
    } else {
      z.long.indx <- 0:(J - 1)
    }
    # y is stored in the following order: species, site, visit
    y <- c(y)
    names.long <- which(!is.na(y))
    # Remove missing observations when the covariate data are available but
    # there are missing detection-nondetection data. 
    if (nrow(X.p) == length(y)) {
      X.p <- X.p[!is.na(y), , drop = FALSE]  
    }
    if (nrow(X.p.re) == length(y) & p.det.re > 0) {
      X.p.re <- X.p.re[!is.na(y), , drop = FALSE]
    }
    y <- y[!is.na(y)]
    # Number of data points for the y vector
    n.obs <- nrow(X.p)

    # Get random effect matrices all set ----------------------------------
    if (p.occ.re > 1) {
      for (j in 2:p.occ.re) {
        X.re[, j] <- X.re[, j] + max(X.re[, j - 1]) + 1
      }
    }
    if (p.det.re > 1) {
      for (j in 2:p.det.re) {
        X.p.re[, j] <- X.p.re[, j] + max(X.p.re[, j - 1]) + 1
      }
    }
    lambda.psi <- matrix(0, J, n.occ.re)
    lambda.p <- matrix(0, n.obs, n.det.re)
    if (p.occ.re > 0) {
      for (i in 1:n.occ.re) {
        lambda.psi[which(X.re == (i - 1), arr.ind = TRUE)[, 1], i] <- 1
      }
    }
    if (p.det.re > 0) {
      for (i in 1:n.det.re) {
        lambda.p[which(X.p.re == (i - 1), arr.ind = TRUE)[, 1], i] <- 1
      }
    }
    # Priors --------------------------------------------------------------
    if (missing(priors)) {
      priors <- list()
    }
    names(priors) <- tolower(names(priors))
    # beta -----------------------
    if ("beta.normal" %in% names(priors)) {
      if (!is.list(priors$beta.normal) | length(priors$beta.normal) != 2) {
        stop("error: beta.normal must be a list of length 2")
      }
      mu.beta <- priors$beta.normal[[1]]
      sigma.beta <- priors$beta.normal[[2]]
      if (length(mu.beta) != p.occ & length(mu.beta) != 1) {
        if (p.occ == 1) {
          stop(paste("error: beta.normal[[1]] must be a vector of length ",
	  	     p.occ, " with elements corresponding to betas' mean", sep = ""))
	} else {
          stop(paste("error: beta.normal[[1]] must be a vector of length ",
	  	     p.occ, " or 1 with elements corresponding to betas' mean", sep = ""))
        }
      }
      if (length(sigma.beta) != p.occ & length(sigma.beta) != 1) {
        if (p.occ == 1) {
          stop(paste("error: beta.normal[[2]] must be a vector of length ",
		   p.occ, " with elements corresponding to betas' variance", sep = ""))
        } else {
          stop(paste("error: beta.normal[[2]] must be a vector of length ",
		   p.occ, " or 1 with elements corresponding to betas' variance", sep = ""))
        }
      }
      if (length(sigma.beta) != p.occ) {
        sigma.beta <- rep(sigma.beta, p.occ)
      }
      if (length(mu.beta) != p.occ) {
        mu.beta <- rep(mu.beta, p.occ)
      }
      Sigma.beta <- sigma.beta * diag(p.occ)
    } else {
      if (verbose) {
        message("No prior specified for beta.normal.\nSetting prior mean to 0 and prior variance to 2.72\n")
      }
      mu.beta <- rep(0, p.occ)
      sigma.beta <- rep(2.72, p.occ)
      Sigma.beta <- diag(p.occ) * 2.72
    }
    # alpha -----------------------
    if ("alpha.normal" %in% names(priors)) {
      if (!is.list(priors$alpha.normal) | length(priors$alpha.normal) != 2) {
        stop("error: alpha.normal must be a list of length 2")
      }
      mu.alpha <- priors$alpha.normal[[1]]
      sigma.alpha <- priors$alpha.normal[[2]]
      if (length(mu.alpha) != p.det & length(mu.alpha) != 1) {
        if (p.det == 1) {
          stop(paste("error: alpha.normal[[1]] must be a vector of length ",
	  	     p.det, " with elements corresponding to alphas' mean", sep = ""))
	} else {
          stop(paste("error: alpha.normal[[1]] must be a vector of length ",
	  	     p.det, " or 1 with elements corresponding to alphas' mean", sep = ""))
        }
      }
      if (length(sigma.alpha) != p.det & length(sigma.alpha) != 1) {
        if (p.det == 1) {
          stop(paste("error: alpha.normal[[2]] must be a vector of length ",
		   p.det, " with elements corresponding to alphas' variance", sep = ""))
        } else {
          stop(paste("error: alpha.normal[[2]] must be a vector of length ",
		   p.det, " or 1 with elements corresponding to alphas' variance", sep = ""))
        }
      }
      if (length(sigma.alpha) != p.det) {
        sigma.alpha <- rep(sigma.alpha, p.det)
      }
      if (length(mu.alpha) != p.det) {
        mu.alpha <- rep(mu.alpha, p.det)
      }
      Sigma.alpha <- sigma.alpha * diag(p.det)
    } else {
      if (verbose) {
        message("No prior specified for alpha.normal.\nSetting prior mean to 0 and prior variance to 2.72\n")
      }
      mu.alpha <- rep(0, p.det)
      sigma.alpha <- rep(2.72, p.det)
      Sigma.alpha <- diag(p.det) * 2.72
    }
    # sigma.sq.psi --------------------
    if (p.occ.re > 0) {
      if ("sigma.sq.psi.ig" %in% names(priors)) {
        if (!is.list(priors$sigma.sq.psi.ig) | length(priors$sigma.sq.psi.ig) != 2) {
          stop("error: sigma.sq.psi.ig must be a list of length 2")
        }
        sigma.sq.psi.a <- priors$sigma.sq.psi.ig[[1]]
        sigma.sq.psi.b <- priors$sigma.sq.psi.ig[[2]]
        if (length(sigma.sq.psi.a) != p.occ.re & length(sigma.sq.psi.a) != 1) {
          if (p.occ.re == 1) {
          stop(paste("error: sigma.sq.psi.ig[[1]] must be a vector of length ", 
          	   p.occ.re, " with elements corresponding to sigma.sq.psis' shape", sep = ""))
	  } else {
          stop(paste("error: sigma.sq.psi.ig[[1]] must be a vector of length ", 
          	   p.occ.re, " or 1 with elements corresponding to sigma.sq.psis' shape", sep = ""))
          }
        }
        if (length(sigma.sq.psi.b) != p.occ.re & length(sigma.sq.psi.b) != 1) {
          if (p.occ.re == 1) {
            stop(paste("error: sigma.sq.psi.ig[[2]] must be a vector of length ", 
          	   p.occ.re, " with elements corresponding to sigma.sq.psis' scale", sep = ""))
	  } else {
            stop(paste("error: sigma.sq.psi.ig[[2]] must be a vector of length ", 
          	   p.occ.re, " or 1with elements corresponding to sigma.sq.psis' scale", sep = ""))
          }
        }
	if (length(sigma.sq.psi.a) != p.occ.re) {
          sigma.sq.psi.a <- rep(sigma.sq.psi.a, p.occ.re)
        }
	if (length(sigma.sq.psi.b) != p.occ.re) {
          sigma.sq.psi.b <- rep(sigma.sq.psi.b, p.occ.re)
        }
    }   else {
        if (verbose) {	    
          message("No prior specified for sigma.sq.psi.ig.\nSetting prior shape to 2 and prior scale to 1\n")
        }
        sigma.sq.psi.a <- rep(2, p.occ.re)
        sigma.sq.psi.b <- rep(1, p.occ.re)
      }
    }
    # sigma.sq.p --------------------
    if (p.det.re > 0) {
      if ("sigma.sq.p.ig" %in% names(priors)) {
        if (!is.list(priors$sigma.sq.p.ig) | length(priors$sigma.sq.p.ig) != 2) {
          stop("error: sigma.sq.p.ig must be a list of length 2")
        }
        sigma.sq.p.a <- priors$sigma.sq.p.ig[[1]]
        sigma.sq.p.b <- priors$sigma.sq.p.ig[[2]]
        if (length(sigma.sq.p.a) != p.det.re & length(sigma.sq.p.a) != 1) {
          if (p.det.re == 1) {
            stop(paste("error: sigma.sq.p.ig[[1]] must be a vector of length ", 
          	   p.det.re, " with elements corresponding to sigma.sq.ps' shape", sep = ""))
	  } else {
            stop(paste("error: sigma.sq.p.ig[[1]] must be a vector of length ", 
          	   p.det.re, " or 1 with elements corresponding to sigma.sq.ps' shape", sep = ""))
          }
        }
        if (length(sigma.sq.p.b) != p.det.re & length(sigma.sq.p.b) != 1) {
          if (p.det.re == 1) {
            stop(paste("error: sigma.sq.p.ig[[2]] must be a vector of length ", 
          	     p.det.re, " with elements corresponding to sigma.sq.ps' scale", sep = ""))
	  } else {
            stop(paste("error: sigma.sq.p.ig[[2]] must be a vector of length ", 
          	     p.det.re, " or 1 with elements corresponding to sigma.sq.ps' scale", sep = ""))
          }
        }
	if (length(sigma.sq.p.a) != p.det.re) {
          sigma.sq.p.a <- rep(sigma.sq.p.a, p.det.re)
        }
	if (length(sigma.sq.p.b) != p.det.re) {
          sigma.sq.p.b <- rep(sigma.sq.p.b, p.det.re)
        }
    }   else {
        if (verbose) {	    
          message("No prior specified for sigma.sq.p.ig.\nSetting prior shape to 0.1 and prior scale to 0.1\n")
        }
        sigma.sq.p.a <- rep(0.1, p.det.re)
        sigma.sq.p.b <- rep(0.1, p.det.re)
      }
    }

    # Starting values -----------------------------------------------------
    if (missing(inits)) {
      inits <- list()
    }
    names(inits) <- tolower(names(inits))
    # z -------------------------------
    if ("z" %in% names(inits)) {
      z.inits <- inits$z
      if (!is.vector(z.inits)) {
        stop(paste("error: initial values for z must be a vector of length ",
		   J, sep = ""))
      }
      if (length(z.inits) != J) {
        stop(paste("error: initial values for z must be a vector of length ",
		   J, sep = ""))
      }
      z.test <- apply(y.big, 1, max, na.rm = TRUE)
      init.test <- sum(z.inits < z.test)
      if (init.test > 0) {
        stop("error: initial values for latent occurrence (z) are invalid. Please re-specify inits$z so initial values are 1 if the species is observed at that site.")
      }
    } else {
      # In correct order since you reordered y.
      z.inits <- apply(y.big, 1, max, na.rm = TRUE)
      if (verbose) {
        message("z is not specified in initial values.\nSetting initial values based on observed data\n")
      }
      
    }
    # beta -----------------------
    if ("beta" %in% names(inits)) {
      beta.inits <- inits[["beta"]]
      if (length(beta.inits) != p.occ & length(beta.inits) != 1) {
        if (p.occ == 1) {
          stop(paste("error: initial values for beta must be of length ", p.occ,
		     sep = ""))

        } else {
          stop(paste("error: initial values for beta must be of length ", p.occ, " or 1",
	  	     sep = ""))
        }
      }
      if (length(beta.inits) != p.occ) {
        beta.inits <- rep(beta.inits, p.occ)
      }
    } else {
      beta.inits <- rnorm(p.occ, mu.beta, sqrt(sigma.beta))
      if (verbose) {
        message('beta is not specified in initial values.\nSetting initial values to random values from the prior distribution\n')
      }
    }
    # alpha -----------------------
    if ("alpha" %in% names(inits)) {
      alpha.inits <- inits[["alpha"]]
      if (length(alpha.inits) != p.det & length(alpha.inits) != 1) {
        if (p.det == 1) {
        stop(paste("error: initial values for alpha must be of length ", p.det,
		   sep = ""))
	} else {
          stop(paste("error: initial values for alpha must be of length ", p.det, " or 1",
		     sep = ""))
        }
      }
      if (length(alpha.inits) != p.det) {
        alpha.inits <- rep(alpha.inits, p.det)
      }
    } else {
      alpha.inits <- rnorm(p.det, mu.alpha, sqrt(sigma.alpha))
      if (verbose) {
        message("alpha is not specified in initial values.\nSetting initial values to random values from the prior distribution\n")
      }
    }

    # sigma.sq.psi -------------------
    if (p.occ.re > 0) {
      if ("sigma.sq.psi" %in% names(inits)) {
        sigma.sq.psi.inits <- inits[["sigma.sq.psi"]]
        if (length(sigma.sq.psi.inits) != p.occ.re & length(sigma.sq.psi.inits) != 1) {
          if (p.occ.re == 1) {
            stop(paste("error: initial values for sigma.sq.psi must be of length ", p.occ.re, 
		       sep = ""))
	  } else {
            stop(paste("error: initial values for sigma.sq.psi must be of length ", p.occ.re, 
		       " or 1", sep = ""))
          }
        }
	if (length(sigma.sq.psi.inits) != p.occ.re) {
          sigma.sq.psi.inits <- rep(sigma.sq.psi.inits, p.occ.re)
        }
      } else {
        sigma.sq.psi.inits <- runif(p.occ.re, 0.5, 10)
        if (verbose) {
          message("sigma.sq.psi is not specified in initial values.\nSetting initial values to random values between 0.5 and 10\n")
        }
      }
      beta.star.indx <- rep(0:(p.occ.re - 1), n.occ.re.long)
      beta.star.inits <- rnorm(n.occ.re, sqrt(sigma.sq.psi.inits[beta.star.indx + 1]))
    }
    # sigma.sq.p ------------------
    if (p.det.re > 0) {
      if ("sigma.sq.p" %in% names(inits)) {
        sigma.sq.p.inits <- inits[["sigma.sq.p"]]
        if (length(sigma.sq.p.inits) != p.det.re & length(sigma.sq.p.inits) != 1) {
          if (p.det.re == 1) {
            stop(paste("error: initial values for sigma.sq.p must be of length ", p.det.re, 
		     sep = ""))
	  } else {
            stop(paste("error: initial values for sigma.sq.p must be of length ", p.det.re, 
		       " or 1", sep = ""))
            
          }
        }
	if (length(sigma.sq.p.inits) != p.det.re) {
          sigma.sq.p.inits <- rep(sigma.sq.p.inits, p.det.re)
        }
      } else {
        sigma.sq.p.inits <- runif(p.det.re, 0.5, 10)
        if (verbose) {
          message("sigma.sq.p is not specified in initial values.\nSetting initial values to random values between 0.5 and 10\n")
        }
      }
      alpha.star.indx <- rep(0:(p.det.re - 1), n.det.re.long)
      alpha.star.inits <- rnorm(n.det.re, sqrt(sigma.sq.p.inits[alpha.star.indx + 1]))
    }

    # Set storage for all variables ---------------------------------------
    storage.mode(y) <- "double"
    storage.mode(z.inits) <- "double"
    storage.mode(X.p) <- "double"
    storage.mode(X) <- "double"
    storage.mode(p.det) <- "integer"
    storage.mode(p.occ) <- "integer"
    storage.mode(J) <- "integer"
    storage.mode(n.obs) <- "integer"
    storage.mode(K) <- "double"
    storage.mode(beta.inits) <- "double"
    storage.mode(alpha.inits) <- "double"
    storage.mode(z.long.indx) <- "integer"
    storage.mode(mu.beta) <- "double"
    storage.mode(Sigma.beta) <- "double"
    storage.mode(mu.alpha) <- "double"
    storage.mode(Sigma.alpha) <- "double"
    storage.mode(n.samples) <- "integer"
    storage.mode(n.omp.threads) <- "integer"
    storage.mode(verbose) <- "integer"
    storage.mode(n.report) <- "integer"
    storage.mode(n.burn) <- "integer"
    storage.mode(n.thin) <- "integer"
    n.post.samples <- length(seq(from = n.burn + 1, 
				 to = n.samples, 
				 by = as.integer(n.thin)))
    storage.mode(n.post.samples) <- "integer"
    # Set model.deviance to NA for returning when no cross-validation
    model.deviance <- NA

    if (p.occ.re > 0 & p.det.re == 0) {

      storage.mode(p.occ.re) <- "integer"
      storage.mode(X.re) <- "integer"
      storage.mode(n.occ.re) <- "integer"
      storage.mode(n.occ.re.long) <- "integer"
      storage.mode(sigma.sq.psi.inits) <- "double"
      storage.mode(sigma.sq.psi.a) <- "double"
      storage.mode(sigma.sq.psi.b) <- "double"
      storage.mode(beta.star.inits) <- "double"
      storage.mode(beta.star.indx) <- "integer"
      storage.mode(lambda.psi) <- "double"
      

      out <- .Call("PGOccREOcc", y, X, X.p, X.re,  
		   lambda.psi, p.occ, p.det, 
		   p.occ.re, J, n.obs, K, n.occ.re,  
		   n.occ.re.long, beta.inits, 
		   alpha.inits, sigma.sq.psi.inits, beta.star.inits, 
		   z.inits, z.long.indx, beta.star.indx, 
		   mu.beta, mu.alpha, Sigma.beta, Sigma.alpha, 
		   sigma.sq.psi.a, sigma.sq.psi.b, n.samples,
          	   n.omp.threads, verbose, n.report, n.burn, n.thin, 
          	   n.post.samples)


      out$beta.samples <- mcmc(t(out$beta.samples))
      colnames(out$beta.samples) <- x.names
      out$alpha.samples <- mcmc(t(out$alpha.samples))
      colnames(out$alpha.samples) <- x.p.names
      out$z.samples <- mcmc(t(out$z.samples))
      out$psi.samples <- mcmc(t(out$psi.samples))
      out$sigma.sq.psi.samples <- mcmc(t(out$sigma.sq.psi.samples))
      colnames(out$sigma.sq.psi.samples) <- x.re.names
      out$beta.star.samples <- mcmc(t(out$beta.star.samples))
      tmp.names <- unlist(sapply(n.occ.re.long, function(a) 1:a))
      beta.star.names <- paste(rep(x.re.names, n.occ.re.long), tmp.names, sep = '-')
      colnames(out$beta.star.samples) <- beta.star.names
      out$X <- X
      out$X.p <- X.p
      out$X.re <- X.re
      out$y <- y.big
      out$n.samples <- n.samples
      out$call <- cl
      out$n.post <- n.post.samples
      out$n.thin <- n.thin
      out$n.burn <- n.burn
      out$pRE <- FALSE
      out$psiRE <- TRUE

      # K-fold cross-validation -------
      if (!missing(k.fold)) {
        if (verbose) {
          cat("----------------------------------------\n");
          cat("\tCross-validation\n");
          cat("----------------------------------------\n");
          message(paste("Performing ", k.fold, "-fold cross-validation using ", k.fold.threads,
	  	        " thread(s).", sep = ''))
	}
	set.seed(k.fold.seed)
	# Number of sites in each hold out data set. 
	sites.random <- sample(1:J)    
        sites.k.fold <- split(sites.random, sites.random %% k.fold)
	registerDoParallel(k.fold.threads)
	model.deviance <- foreach (i = 1:k.fold, .combine = sum) %dopar% {
          curr.set <- sort(sites.random[sites.k.fold[[i]]])
          if (binom) {
            y.indx <- !(1:J %in% curr.set)
          } else {
	    y.indx <- !((z.long.indx + 1) %in% curr.set)
          }
          y.fit <- y[y.indx]
	  y.0 <- y[!y.indx]
	  y.big.fit <- y.big[-curr.set, , drop = FALSE]
	  y.big.0 <- y.big[curr.set, , drop = FALSE]
	  z.inits.fit <- z.inits[-curr.set]
	  X.p.fit <- X.p[y.indx, , drop = FALSE]
	  X.p.0 <- X.p[!y.indx, , drop = FALSE]
	  X.fit <- X[-curr.set, , drop = FALSE]
	  X.0 <- X[curr.set, , drop = FALSE]
	  J.fit <- nrow(X.fit)
	  J.0 <- nrow(X.0)
	  K.fit <- K[-curr.set]
	  K.0 <- K[curr.set]
	  n.obs.fit <- nrow(X.p.fit)
	  n.obs.0 <- nrow(X.p.0)
	  lambda.psi.fit <- lambda.psi[-curr.set, , drop = FALSE]
	  lambda.psi.0 <- lambda.psi[curr.set, , drop = FALSE]
	  X.re.fit <- X.re[-curr.set, , drop = FALSE]
	  X.re.0 <- X.re[curr.set, , drop = FALSE]
	  # Gotta be a better way, but will do for now. 
	  if (binom) {
            z.long.indx.fit <- 0:(J.fit - 1)
            z.0.long.indx <- 1:J.0
          } else {
	    z.long.indx.fit <- matrix(NA, J.fit, max(K.fit))
	    for (j in 1:J.fit) {
              z.long.indx.fit[j, 1:K.fit[j]] <- j  
            }
	    z.long.indx.fit <- c(z.long.indx.fit)
	    z.long.indx.fit <- z.long.indx.fit[!is.na(z.long.indx.fit)] - 1
	    z.0.long.indx <- matrix(NA, nrow(X.0), max(K.0))
	    for (j in 1:nrow(X.0)) {
              z.0.long.indx[j, 1:K.0[j]] <- j  
            }
	    z.0.long.indx <- c(z.0.long.indx)
	    z.0.long.indx <- z.0.long.indx[!is.na(z.0.long.indx)] 
	  }
	  verbose.fit <- FALSE
	  n.omp.threads.fit <- 1

          storage.mode(y.fit) <- "double"
          storage.mode(z.inits.fit) <- "double"
          storage.mode(X.p.fit) <- "double"
          storage.mode(X.fit) <- "double"
          storage.mode(p.det) <- "integer"
          storage.mode(p.occ) <- "integer"
          storage.mode(J.fit) <- "integer"
          storage.mode(n.obs.fit) <- "integer"
          storage.mode(K.fit) <- "double"
          storage.mode(beta.inits) <- "double"
          storage.mode(alpha.inits) <- "double"
          storage.mode(z.long.indx.fit) <- "integer"
          storage.mode(mu.beta) <- "double"
          storage.mode(Sigma.beta) <- "double"
          storage.mode(mu.alpha) <- "double"
          storage.mode(Sigma.alpha) <- "double"
          storage.mode(n.samples) <- "integer"
          storage.mode(n.omp.threads.fit) <- "integer"
          storage.mode(verbose.fit) <- "integer"
          storage.mode(n.report) <- "integer"
          storage.mode(n.burn) <- "integer"
          storage.mode(n.thin) <- "integer"
          storage.mode(p.occ.re) <- "integer"
          storage.mode(X.re.fit) <- "integer"
          storage.mode(n.occ.re) <- "integer"
          storage.mode(n.occ.re.long) <- "integer"
          storage.mode(sigma.sq.psi.inits) <- "double"
          storage.mode(sigma.sq.psi.a) <- "double"
          storage.mode(sigma.sq.psi.b) <- "double"
          storage.mode(beta.star.inits) <- "double"
          storage.mode(beta.star.indx) <- "integer"
          storage.mode(lambda.psi.fit) <- "double"

          # Run the model in C
          out.fit <- .Call("PGOccREOcc", y.fit, X.fit, X.p.fit, X.re.fit,  
		           lambda.psi.fit, p.occ, p.det, 
		           p.occ.re, J.fit, n.obs.fit, K.fit, n.occ.re,  
		           n.occ.re.long, beta.inits, 
		           alpha.inits, sigma.sq.psi.inits, beta.star.inits, 
		           z.inits.fit, z.long.indx.fit, beta.star.indx, 
		           mu.beta, mu.alpha, Sigma.beta, Sigma.alpha, 
		           sigma.sq.psi.a, sigma.sq.psi.b, n.samples,
          	           n.omp.threads.fit, verbose.fit, n.report, n.burn, n.thin, 
          	           n.post.samples)

	  # Predict occurrence at new sites. 
          psi.0.samples <- mcmc(logit.inv(t(X.0 %*% out.fit$beta.samples + 
					  lambda.psi.0 %*% out.fit$beta.star.samples)))
	  z.0.samples <- matrix(NA, n.post.samples, nrow(X.0))
	  for (j in 1:nrow(X.0)) {
            z.0.samples[, j] <- rbinom(n.post.samples, 1, psi.0.samples[, j])
          }
	  # Detection 
	  p.0.samples <- logit.inv(t(X.p.0 %*% out.fit$alpha.samples))
	  if (binom) {
            like.samples <- matrix(NA, nrow(y.big.0), ncol(y.big.0))
            for (j in 1:nrow(X.p.0)) {
              for (k in 1:K.0[j]) {
                like.samples[j, k] <- mean(dbinom(y.big.0[j, k], 1,
					         p.0.samples[, j] * z.0.samples[, z.0.long.indx[j]]))
	      }
            }
          } else {
	    like.samples <- rep(NA, nrow(X.p.0))
	    for (j in 1:nrow(X.p.0)) {
              like.samples[j] <- mean(dbinom(y.0[j], 1, p.0.samples[, j] * z.0.samples[, z.0.long.indx[j]]))
            }
	  }
	  sum(log(like.samples), na.rm = TRUE)
        }
	model.deviance <- -2 * model.deviance
	# Return objects from cross-validation
	out$k.fold.deviance <- model.deviance
	stopImplicitCluster()
      }
      class(out) <- "PGOcc"
    }

    if (p.occ.re == 0 & p.det.re > 0) {
      
      storage.mode(p.det.re) <- "integer"
      storage.mode(X.p.re) <- "integer"
      storage.mode(n.det.re) <- "integer"
      storage.mode(n.det.re.long) <- "integer"
      storage.mode(sigma.sq.p.inits) <- "double"
      storage.mode(sigma.sq.p.a) <- "double"
      storage.mode(sigma.sq.p.b) <- "double"
      storage.mode(alpha.star.inits) <- "double"
      storage.mode(alpha.star.indx) <- "integer"
      storage.mode(lambda.p) <- "double"
      
      out <- .Call("PGOccREDet", y, X, X.p, X.p.re, lambda.p, p.occ, p.det, 
		   p.det.re, J, n.obs, K, n.det.re, n.det.re.long, beta.inits, 
		   alpha.inits, sigma.sq.p.inits, 
		   alpha.star.inits, z.inits, z.long.indx, alpha.star.indx, 
		   mu.beta, mu.alpha, Sigma.beta, Sigma.alpha, 
		   sigma.sq.p.a, sigma.sq.p.b, n.samples,
          	   n.omp.threads, verbose, n.report, n.burn, n.thin, 
          	   n.post.samples)

      out$beta.samples <- mcmc(t(out$beta.samples))
      colnames(out$beta.samples) <- x.names
      out$alpha.samples <- mcmc(t(out$alpha.samples))
      colnames(out$alpha.samples) <- x.p.names
      out$z.samples <- mcmc(t(out$z.samples))
      out$psi.samples <- mcmc(t(out$psi.samples))
      out$sigma.sq.p.samples <- mcmc(t(out$sigma.sq.p.samples))
      colnames(out$sigma.sq.p.samples) <- x.p.re.names
      out$alpha.star.samples <- mcmc(t(out$alpha.star.samples))
      tmp.names <- unlist(sapply(n.det.re.long, function(a) 1:a))
      alpha.star.names <- paste(rep(x.p.re.names, n.det.re.long), tmp.names, sep = '-')
      colnames(out$alpha.star.samples) <- alpha.star.names
      out$X <- X
      out$X.p <- X.p
      out$X.p.re <- X.p.re
      out$lambda.p <- lambda.p
      out$y <- y.big
      out$n.samples <- n.samples
      out$call <- cl
      out$n.post <- n.post.samples
      out$n.thin <- n.thin
      out$n.burn <- n.burn
      out$pRE <- TRUE
      out$psiRE <- FALSE

      # K-fold cross-validation -------
      if (!missing(k.fold)) {
        if (verbose) {
          cat("----------------------------------------\n");
          cat("\tCross-validation\n");
          cat("----------------------------------------\n");
          message(paste("Performing ", k.fold, "-fold cross-validation using ", k.fold.threads,
		      " thread(s).", sep = ''))
	}
        # Currently implemented without parellization. 
	set.seed(k.fold.seed)
	# Number of sites in each hold out data set. 
	sites.random <- sample(1:J)    
        sites.k.fold <- split(sites.random, sites.random %% k.fold)
	registerDoParallel(k.fold.threads)
	model.deviance <- foreach (i = 1:k.fold, .combine = sum) %dopar% {
          curr.set <- sort(sites.random[sites.k.fold[[i]]])
          if (binom) {
            y.indx <- !(1:J %in% curr.set)
          } else {
	    y.indx <- !((z.long.indx + 1) %in% curr.set)
          }
          y.fit <- y[y.indx]
	  y.0 <- y[!y.indx]
	  y.big.fit <- y.big[-curr.set, , drop = FALSE]
	  y.big.0 <- y.big[curr.set, , drop = FALSE]
	  z.inits.fit <- z.inits[-curr.set]
	  X.p.fit <- X.p[y.indx, , drop = FALSE]
	  X.p.0 <- X.p[!y.indx, , drop = FALSE]
	  X.fit <- X[-curr.set, , drop = FALSE]
	  X.0 <- X[curr.set, , drop = FALSE]
	  n.obs.fit <- nrow(X.p.fit)
	  n.obs.0 <- nrow(X.p.0)
	  J.fit <- nrow(X.fit)
	  J.0 <- nrow(X.0)
	  K.fit <- K[-curr.set]
	  K.0 <- K[curr.set]
	  lambda.p.fit <- lambda.p[y.indx, , drop = FALSE]
	  lambda.p.0 <- lambda.p[!y.indx, , drop = FALSE]
	  X.p.re.fit <- X.p.re[y.indx, , drop = FALSE]
	  X.p.re.0 <- X.p.re[!y.indx, , drop = FALSE]
	  # Gotta be a better way, but will do for now. 
	  if (binom) {
            z.long.indx.fit <- 0:(J.fit - 1)
            z.0.long.indx <- 1:J.0
          } else {
	    z.long.indx.fit <- matrix(NA, J.fit, max(K.fit))
	    for (j in 1:J.fit) {
              z.long.indx.fit[j, 1:K.fit[j]] <- j  
            }
	    z.long.indx.fit <- c(z.long.indx.fit)
	    z.long.indx.fit <- z.long.indx.fit[!is.na(z.long.indx.fit)] - 1
	    z.0.long.indx <- matrix(NA, nrow(X.0), max(K.0))
	    for (j in 1:nrow(X.0)) {
              z.0.long.indx[j, 1:K.0[j]] <- j  
            }
	    z.0.long.indx <- c(z.0.long.indx)
	    z.0.long.indx <- z.0.long.indx[!is.na(z.0.long.indx)] 
	  }
	  verbose.fit <- FALSE
	  n.omp.threads.fit <- 1

          storage.mode(y.fit) <- "double"
          storage.mode(z.inits.fit) <- "double"
          storage.mode(X.p.fit) <- "double"
          storage.mode(X.fit) <- "double"
          storage.mode(p.det) <- "integer"
          storage.mode(p.occ) <- "integer"
          storage.mode(J.fit) <- "integer"
          storage.mode(n.obs.fit) <- "integer"
          storage.mode(K.fit) <- "double"
          storage.mode(beta.inits) <- "double"
          storage.mode(alpha.inits) <- "double"
          storage.mode(z.long.indx.fit) <- "integer"
          storage.mode(mu.beta) <- "double"
          storage.mode(Sigma.beta) <- "double"
          storage.mode(mu.alpha) <- "double"
          storage.mode(Sigma.alpha) <- "double"
          storage.mode(n.samples) <- "integer"
          storage.mode(n.omp.threads.fit) <- "integer"
          storage.mode(verbose.fit) <- "integer"
          storage.mode(n.report) <- "integer"
          storage.mode(n.burn) <- "integer"
          storage.mode(n.thin) <- "integer"
          storage.mode(p.det.re) <- "integer"
          storage.mode(X.p.re.fit) <- "integer"
          storage.mode(n.det.re) <- "integer"
          storage.mode(n.det.re.long) <- "integer"
          storage.mode(sigma.sq.p.inits) <- "double"
          storage.mode(sigma.sq.p.a) <- "double"
          storage.mode(sigma.sq.p.b) <- "double"
          storage.mode(alpha.star.inits) <- "double"
          storage.mode(alpha.star.indx) <- "integer"
          storage.mode(lambda.p.fit) <- "double"

          # Run the model in C
          out.fit <- .Call("PGOccREDet", y.fit, X.fit, X.p.fit, X.p.re.fit,  
		           lambda.p.fit, p.occ, p.det, 
		           p.det.re, J.fit, n.obs.fit, K.fit, n.det.re,  
		           n.det.re.long, beta.inits, 
		           alpha.inits, sigma.sq.p.inits, alpha.star.inits, 
		           z.inits.fit, z.long.indx.fit, alpha.star.indx, 
		           mu.beta, mu.alpha, Sigma.beta, Sigma.alpha, 
		           sigma.sq.p.a, sigma.sq.p.b, n.samples,
          	           n.omp.threads.fit, verbose.fit, n.report, n.burn, n.thin, 
          	           n.post.samples)

	  # Predict occurrence at new sites. 
          # Now can get the output
          psi.0.samples <- mcmc(logit.inv(t(X.0 %*% out.fit$beta.samples)))
	  z.0.samples <- matrix(NA, n.post.samples, nrow(X.0))
	  for (j in 1:nrow(X.0)) {
            z.0.samples[, j] <- rbinom(n.post.samples, 1, psi.0.samples[, j])
          }
	  # Detection 
	  p.0.samples <- logit.inv(t(X.p.0 %*% out.fit$alpha.samples + 
				   lambda.p.0 %*% out.fit$alpha.star.samples))
	  if (binom) {
            like.samples <- matrix(NA, nrow(y.big.0), ncol(y.big.0))
            for (j in 1:nrow(X.p.0)) {
              for (k in 1:K.0[j]) {
                like.samples[j, k] <- mean(dbinom(y.big.0[j, k], 1,
					         p.0.samples[, j] * z.0.samples[, z.0.long.indx[j]]))
	      }
            }
          } else {
	    like.samples <- rep(NA, nrow(X.p.0))
	    for (j in 1:nrow(X.p.0)) {
              like.samples[j] <- mean(dbinom(y.0[j], 1, p.0.samples[, j] * z.0.samples[, z.0.long.indx[j]]))
            }
	  }
	  sum(log(like.samples), na.rm = TRUE)
        }
	model.deviance <- -2 * model.deviance
	# Return objects from cross-validation
	out$k.fold.deviance <- model.deviance
	stopImplicitCluster()
      }

      class(out) <- "PGOcc"

    }

    if (p.occ.re > 0 & p.det.re > 0) {

      storage.mode(p.occ.re) <- "integer"
      storage.mode(p.det.re) <- "integer"
      storage.mode(X.re) <- "integer"
      storage.mode(X.p.re) <- "integer"
      storage.mode(n.occ.re) <- "integer"
      storage.mode(n.det.re) <- "integer"
      storage.mode(n.occ.re.long) <- "integer"
      storage.mode(n.det.re.long) <- "integer"
      storage.mode(sigma.sq.psi.inits) <- "double"
      storage.mode(sigma.sq.p.inits) <- "double"
      storage.mode(sigma.sq.psi.a) <- "double"
      storage.mode(sigma.sq.psi.b) <- "double"
      storage.mode(sigma.sq.p.a) <- "double"
      storage.mode(sigma.sq.p.b) <- "double"
      storage.mode(beta.star.inits) <- "double"
      storage.mode(beta.star.indx) <- "integer"
      storage.mode(alpha.star.inits) <- "double"
      storage.mode(alpha.star.indx) <- "integer"
      storage.mode(lambda.psi) <- "double"
      storage.mode(lambda.p) <- "double"
      

      out <- .Call("PGOccREBoth", y, X, X.p, X.re, X.p.re, 
		   lambda.psi, lambda.p, p.occ, p.det, 
		   p.occ.re, p.det.re, J, n.obs, K, n.occ.re, n.det.re, 
		   n.occ.re.long, n.det.re.long, beta.inits, 
		   alpha.inits, sigma.sq.psi.inits, 
		   sigma.sq.p.inits, beta.star.inits, 
		   alpha.star.inits, z.inits, z.long.indx, 
		   beta.star.indx, alpha.star.indx, 
		   mu.beta, mu.alpha, Sigma.beta, Sigma.alpha, 
		   sigma.sq.psi.a, sigma.sq.psi.b, 
		   sigma.sq.p.a, sigma.sq.p.b, n.samples,
          	   n.omp.threads, verbose, n.report, n.burn, n.thin, 
          	   n.post.samples)

      out$beta.samples <- mcmc(t(out$beta.samples))
      colnames(out$beta.samples) <- x.names
      out$alpha.samples <- mcmc(t(out$alpha.samples))
      colnames(out$alpha.samples) <- x.p.names
      out$z.samples <- mcmc(t(out$z.samples))
      out$psi.samples <- mcmc(t(out$psi.samples))
      out$sigma.sq.psi.samples <- mcmc(t(out$sigma.sq.psi.samples))
      colnames(out$sigma.sq.psi.samples) <- x.re.names
      out$sigma.sq.p.samples <- mcmc(t(out$sigma.sq.p.samples))
      colnames(out$sigma.sq.p.samples) <- x.p.re.names
      out$beta.star.samples <- mcmc(t(out$beta.star.samples))
      tmp.names <- unlist(sapply(n.occ.re.long, function(a) 1:a))
      beta.star.names <- paste(rep(x.re.names, n.occ.re.long), tmp.names, sep = '-')
      colnames(out$beta.star.samples) <- beta.star.names
      out$alpha.star.samples <- mcmc(t(out$alpha.star.samples))
      tmp.names <- unlist(sapply(n.det.re.long, function(a) 1:a))
      alpha.star.names <- paste(rep(x.p.re.names, n.det.re.long), tmp.names, sep = '-')
      colnames(out$alpha.star.samples) <- alpha.star.names
      out$X <- X
      out$X.p <- X.p
      out$X.re <- X.re
      out$X.p.re <- X.p.re
      out$lambda.p <- lambda.p
      out$y <- y.big
      out$n.samples <- n.samples
      out$call <- cl
      out$n.post <- n.post.samples
      out$n.thin <- n.thin
      out$n.burn <- n.burn
      out$pRE <- TRUE
      out$psiRE <- TRUE


      # K-fold cross-validation -------
      if (!missing(k.fold)) {
        if (verbose) {
          cat("----------------------------------------\n");
          cat("\tCross-validation\n");
          cat("----------------------------------------\n");
          message(paste("Performing ", k.fold, "-fold cross-validation using ", k.fold.threads,
		      " thread(s).", sep = ''))
	}
        # Currently implemented without parellization. 
	set.seed(k.fold.seed)
	# Number of sites in each hold out data set. 
	sites.random <- sample(1:J)    
        sites.k.fold <- split(sites.random, sites.random %% k.fold)
	registerDoParallel(k.fold.threads)
	model.deviance <- foreach (i = 1:k.fold, .combine = sum) %dopar% {
          curr.set <- sort(sites.random[sites.k.fold[[i]]])
          if (binom) {
            y.indx <- !(1:J %in% curr.set)
          } else {
	    y.indx <- !((z.long.indx + 1) %in% curr.set)
          }
          y.fit <- y[y.indx]
	  y.0 <- y[!y.indx]
	  y.big.fit <- y.big[-curr.set, , drop = FALSE]
	  y.big.0 <- y.big[curr.set, , drop = FALSE]
	  z.inits.fit <- z.inits[-curr.set]
	  X.p.fit <- X.p[y.indx, , drop = FALSE]
	  X.p.0 <- X.p[!y.indx, , drop = FALSE]
	  X.fit <- X[-curr.set, , drop = FALSE]
	  X.0 <- X[curr.set, , drop = FALSE]
	  J.fit <- nrow(X.fit)
	  J.0 <- nrow(X.0)
	  K.fit <- K[-curr.set]
	  K.0 <- K[curr.set]
	  n.obs.fit <- nrow(X.p.fit)
	  n.obs.0 <- nrow(X.p.0)
	  lambda.psi.fit <- lambda.psi[-curr.set, , drop = FALSE]
	  lambda.psi.0 <- lambda.psi[curr.set, , drop = FALSE]
	  X.re.fit <- X.re[-curr.set, , drop = FALSE]
	  X.re.0 <- X.re[curr.set, , drop = FALSE]
	  lambda.p.fit <- lambda.p[y.indx, , drop = FALSE]
	  lambda.p.0 <- lambda.p[!y.indx, , drop = FALSE]
	  X.p.re.fit <- X.p.re[y.indx, , drop = FALSE]
	  X.p.re.0 <- X.p.re[!y.indx, , drop = FALSE]
	  # Gotta be a better way, but will do for now. 
	  if (binom) {
            z.long.indx.fit <- 0:(J.fit - 1)
            z.0.long.indx <- 1:J.0
          } else {
	    z.long.indx.fit <- matrix(NA, J.fit, max(K.fit))
	    for (j in 1:J.fit) {
              z.long.indx.fit[j, 1:K.fit[j]] <- j  
            }
	    z.long.indx.fit <- c(z.long.indx.fit)
	    z.long.indx.fit <- z.long.indx.fit[!is.na(z.long.indx.fit)] - 1
	    z.0.long.indx <- matrix(NA, nrow(X.0), max(K.0))
	    for (j in 1:nrow(X.0)) {
              z.0.long.indx[j, 1:K.0[j]] <- j  
            }
	    z.0.long.indx <- c(z.0.long.indx)
	    z.0.long.indx <- z.0.long.indx[!is.na(z.0.long.indx)] 
	  }
	  verbose.fit <- FALSE
	  n.omp.threads.fit <- 1

          storage.mode(y.fit) <- "double"
          storage.mode(z.inits.fit) <- "double"
          storage.mode(X.p.fit) <- "double"
          storage.mode(X.fit) <- "double"
          storage.mode(p.det) <- "integer"
          storage.mode(p.occ) <- "integer"
          storage.mode(J.fit) <- "integer"
          storage.mode(n.obs.fit) <- "integer"
          storage.mode(K.fit) <- "double"
          storage.mode(beta.inits) <- "double"
          storage.mode(alpha.inits) <- "double"
          storage.mode(z.long.indx.fit) <- "integer"
          storage.mode(mu.beta) <- "double"
          storage.mode(Sigma.beta) <- "double"
          storage.mode(mu.alpha) <- "double"
          storage.mode(Sigma.alpha) <- "double"
          storage.mode(n.samples) <- "integer"
          storage.mode(n.omp.threads.fit) <- "integer"
          storage.mode(verbose.fit) <- "integer"
          storage.mode(n.report) <- "integer"
          storage.mode(n.burn) <- "integer"
          storage.mode(n.thin) <- "integer"
          storage.mode(p.det.re) <- "integer"
          storage.mode(X.p.re.fit) <- "integer"
          storage.mode(n.det.re) <- "integer"
          storage.mode(n.det.re.long) <- "integer"
          storage.mode(sigma.sq.p.inits) <- "double"
          storage.mode(sigma.sq.p.a) <- "double"
          storage.mode(sigma.sq.p.b) <- "double"
          storage.mode(alpha.star.inits) <- "double"
          storage.mode(alpha.star.indx) <- "integer"
          storage.mode(lambda.p.fit) <- "double"
          storage.mode(sigma.sq.psi.inits) <- "double"
          storage.mode(sigma.sq.psi.a) <- "double"
          storage.mode(sigma.sq.psi.b) <- "double"
          storage.mode(beta.star.inits) <- "double"
          storage.mode(beta.star.indx) <- "integer"
          storage.mode(lambda.psi.fit) <- "double"

          # Run the model in C
          out.fit <- .Call("PGOccREBoth", y.fit, X.fit, X.p.fit, X.re.fit, X.p.re.fit, 
		           lambda.psi.fit, lambda.p.fit, p.occ, p.det, 
		           p.occ.re, p.det.re, J.fit, n.obs.fit, K.fit, n.occ.re, n.det.re, 
		           n.occ.re.long, n.det.re.long, beta.inits, 
		           alpha.inits, sigma.sq.psi.inits, 
		           sigma.sq.p.inits, beta.star.inits, 
		           alpha.star.inits, z.inits.fit, z.long.indx.fit, 
		           beta.star.indx, alpha.star.indx, 
		           mu.beta, mu.alpha, Sigma.beta, Sigma.alpha, 
		           sigma.sq.psi.a, sigma.sq.psi.b, 
		           sigma.sq.p.a, sigma.sq.p.b, n.samples,
          	           n.omp.threads.fit, verbose.fit, n.report, n.burn, n.thin, 
          	           n.post.samples)

	  # Predict occurrence at new sites. 
          # Now can get the output
          psi.0.samples <- mcmc(logit.inv(t(X.0 %*% out.fit$beta.samples + 
					  lambda.psi.0 %*% out.fit$beta.star.samples)))
	  z.0.samples <- matrix(NA, n.post.samples, nrow(X.0))
	  for (j in 1:nrow(X.0)) {
            z.0.samples[, j] <- rbinom(n.post.samples, 1, psi.0.samples[, j])
          }
	  p.0.samples <- logit.inv(t(X.p.0 %*% out.fit$alpha.samples + 
				     lambda.p.0 %*% out.fit$alpha.star.samples))
	  if (binom) {
            like.samples <- matrix(NA, nrow(y.big.0), ncol(y.big.0))
            for (j in 1:nrow(X.p.0)) {
              for (k in 1:K.0[j]) {
                like.samples[j, k] <- mean(dbinom(y.big.0[j, k], 1,
					         p.0.samples[, j] * z.0.samples[, z.0.long.indx[j]]))
	      }
            }
          } else {
	    like.samples <- rep(NA, nrow(X.p.0))
	    for (j in 1:nrow(X.p.0)) {
              like.samples[j] <- mean(dbinom(y.0[j], 1, p.0.samples[, j] * z.0.samples[, z.0.long.indx[j]]))
            }
	  }
	  sum(log(like.samples), na.rm = TRUE)
        }
	model.deviance <- -2 * model.deviance
	# Return objects from cross-validation
	out$k.fold.deviance <- model.deviance
	stopImplicitCluster()
      }

      class(out) <- "PGOcc"
      
    }

    if (p.occ.re == 0 & p.det.re == 0) {

      # Run the model in C
      out <- .Call("PGOcc", y, X, X.p, p.occ, p.det, J, n.obs, K,
          	 beta.inits, alpha.inits, z.inits,
          	 z.long.indx, mu.beta, mu.alpha,
          	 Sigma.beta, Sigma.alpha, n.samples,
          	 n.omp.threads, verbose, n.report, n.burn, n.thin, 
          	 n.post.samples)

      out$beta.samples <- mcmc(t(out$beta.samples))
      colnames(out$beta.samples) <- x.names
      out$alpha.samples <- mcmc(t(out$alpha.samples))
      colnames(out$alpha.samples) <- x.p.names
      out$z.samples <- mcmc(t(out$z.samples))
      out$psi.samples <- mcmc(t(out$psi.samples))
      out$X <- X
      out$X.p <- X.p
      out$y <- y.big
      out$n.samples <- n.samples
      out$call <- cl
      out$n.post <- n.post.samples
      out$n.thin <- n.thin
      out$n.burn <- n.burn
      out$pRE <- FALSE
      out$psiRE <- FALSE

      # K-fold cross-validation -------
      if (!missing(k.fold)) {
	if (verbose) {      
          cat("----------------------------------------\n");
          cat("\tCross-validation\n");
          cat("----------------------------------------\n");
          message(paste("Performing ", k.fold, "-fold cross-validation using ", k.fold.threads,
		      " thread(s).", sep = ''))
	}
	set.seed(k.fold.seed)
	# Number of sites in each hold out data set. 
	sites.random <- sample(1:J)    
        sites.k.fold <- split(sites.random, sites.random %% k.fold)
	registerDoParallel(k.fold.threads)
	model.deviance <- foreach (i = 1:k.fold, .combine = sum) %dopar% {
          curr.set <- sort(sites.random[sites.k.fold[[i]]])
          if (binom) {
            y.indx <- !(1:J %in% curr.set)
          } else {
	    y.indx <- !((z.long.indx + 1) %in% curr.set)
          }
          y.fit <- y[y.indx]
	  y.big.fit <- y.big[-curr.set, , drop = FALSE]
	  y.big.0 <- y.big[curr.set, , drop = FALSE]
	  y.0 <- y[!y.indx]
	  z.inits.fit <- z.inits[-curr.set]
	  X.p.fit <- X.p[y.indx, , drop = FALSE]
	  X.p.0 <- X.p[!y.indx, , drop = FALSE]
	  n.obs.fit <- nrow(X.p.fit)
	  n.obs.0 <- nrow(X.p.0)
	  X.fit <- X[-curr.set, , drop = FALSE]
	  X.0 <- X[curr.set, , drop = FALSE]
	  J.fit <- nrow(X.fit)
	  J.0 <- nrow(X.0)
	  K.fit <- K[-curr.set]
	  K.0 <- K[curr.set]
	  # Gotta be a better way, but will do for now. 
	  if (binom) {
            z.long.indx.fit <- 0:(J.fit - 1)
            z.0.long.indx <- 1:J.0
          } else {
	    z.long.indx.fit <- matrix(NA, J.fit, max(K.fit))
	    for (j in 1:J.fit) {
              z.long.indx.fit[j, 1:K.fit[j]] <- j  
            }
	    z.long.indx.fit <- c(z.long.indx.fit)
	    z.long.indx.fit <- z.long.indx.fit[!is.na(z.long.indx.fit)] - 1
	    z.0.long.indx <- matrix(NA, nrow(X.0), max(K.0))
	    for (j in 1:nrow(X.0)) {
              z.0.long.indx[j, 1:K.0[j]] <- j  
            }
	    z.0.long.indx <- c(z.0.long.indx)
	    z.0.long.indx <- z.0.long.indx[!is.na(z.0.long.indx)] 
	  }
	  verbose.fit <- FALSE
	  n.omp.threads.fit <- 1

          storage.mode(y.fit) <- "double"
          storage.mode(z.inits.fit) <- "double"
          storage.mode(X.p.fit) <- "double"
          storage.mode(X.fit) <- "double"
          storage.mode(p.det) <- "integer"
          storage.mode(p.occ) <- "integer"
          storage.mode(J.fit) <- "integer"
          storage.mode(n.obs.fit) <- "integer"
          storage.mode(K.fit) <- "double"
          storage.mode(beta.inits) <- "double"
          storage.mode(alpha.inits) <- "double"
          storage.mode(z.long.indx.fit) <- "integer"
          storage.mode(mu.beta) <- "double"
          storage.mode(Sigma.beta) <- "double"
          storage.mode(mu.alpha) <- "double"
          storage.mode(Sigma.alpha) <- "double"
          storage.mode(n.samples) <- "integer"
          storage.mode(n.omp.threads.fit) <- "integer"
          storage.mode(verbose.fit) <- "integer"
          storage.mode(n.report) <- "integer"
          storage.mode(n.burn) <- "integer"
          storage.mode(n.thin) <- "integer"

          # Run the model in C
          out.fit <- .Call("PGOcc", y.fit, X.fit, X.p.fit, p.occ, p.det, J.fit, n.obs.fit, 
			   K.fit, beta.inits, alpha.inits, z.inits.fit,
              	           z.long.indx.fit, mu.beta, mu.alpha,
              	           Sigma.beta, Sigma.alpha, n.samples,
              	           n.omp.threads.fit, verbose.fit, n.report, n.burn, n.thin, 
              	           n.post.samples)

	  # Predict occurrence at new sites. 
	  psi.0.samples <- logit.inv(t(X.0 %*% out.fit$beta.samples))
	  # Check this if you encounter a bug. 
	  z.0.samples <- matrix(NA, n.post.samples, nrow(X.0))
	  for (j in 1:nrow(X.0)) {
            z.0.samples[, j] <- rbinom(n.post.samples, 1, psi.0.samples[, j])
          }
	  # Detection 
	  p.0.samples <- logit.inv(t(X.p.0 %*% out.fit$alpha.samples))
	  if (binom) {
            like.samples <- matrix(NA, nrow(y.big.0), ncol(y.big.0))
            for (j in 1:nrow(X.p.0)) {
              for (k in 1:K.0[j]) {
                like.samples[j, k] <- mean(dbinom(y.big.0[j, k], 1,
					         p.0.samples[, j] * z.0.samples[, z.0.long.indx[j]]))
	      }
            }
          } else {
	    like.samples <- rep(NA, nrow(X.p.0))
	    for (j in 1:nrow(X.p.0)) {
              like.samples[j] <- mean(dbinom(y.0[j], 1, p.0.samples[, j] * z.0.samples[, z.0.long.indx[j]]))
            }
	  }
	  sum(log(like.samples), na.rm = TRUE)
        }
	model.deviance <- -2 * model.deviance
	# Return objects from cross-validation
	out$k.fold.deviance <- model.deviance
	stopImplicitCluster()
      }

      class(out) <- "PGOcc"
    }
    out$run.time <- proc.time() - ptm
    out
}
