## Copyright (C) 2013 Michael Braun

## n - number of draws
## x - matrix of quantiles, one sample per ROW
## mu - vector for mean (same for each sample)
## CH - object of class CHMsimpl or CHM super, as
##      generated by the Cholesky function in the
##      Matrix package.  Represents the Cholesky
##      decomposition of either the covariance or
##      precision matrix.
## prec - logical value if CH is the decomposition
##        of the precision matrix (TRUE) or covariance
##        matrix (FALSE)  Default is TRUE.
##        However, if covariance is sparse, rmvn.sparse
##        may run more quickly using that instead of the
##        precision matrix.  That said, if the precision
##        matrix is sparse, the corresponding covariance
##        matrix may not be.


rmvn.sparse <- function(n, mu, CH, prec=TRUE) {

  if (is.na(match(class(CH),c("dCHMsimpl","dCHMsuper")))) {
    stop("CH must be an object of class 'dCHMsimpl' or 'dCHMsuper'")
  }

  k <- length(mu)

  if (!(k>0)) {
    stop("mu must have positive length")
  }
  if (!(n>0)) {
    stop("n must be positive")
  }

  if (!(k==dim(CH)[1])) {
    stop("dimensions of mu and CH do not conform")
  }

  if (!is.logical(prec)) {
      stop("prec must be either TRUE or FALSE")
  }

  x <- rnorm(n*k)
  dim(x) <- c(k,n)

  A <- expand(CH)

  if (prec) {
    y <- solve(t(A$L),x) ## L'y = x
  } else {
    y <- A$L %*% x
  }
  
  y <- as(crossprod(A$P,y),"matrix") ## P' %*% y

  y <- y + mu
  
  return(t(y))
    
}

dmvn.sparse <- function(x, mu, CH, prec=TRUE) {

  
  if (is.vector(x)) x <- matrix(x,nrow=1)
    
  k <- length(mu)
  n <- NROW(x)
  
  if (!(k>0)) {
    stop("mu must have positive length")
  }
  
  if (!(k==dim(CH)[1])) {
    stop("dimensions of mu and CH do not conform")
  }

  if (k!=NCOL(x)) {
    stop("x must have same number of columns as the length of mu")
  }

  
  if (!is.logical(prec)) {
      stop("prec must be either TRUE or FALSE")
  }
 
  A <- expand(CH)
  detL <- sum(log(diag(A$L)))
  C <- -0.918938533204672669541*k ## -k*log(2*pi)/2
  
  
  xmu <- t(x)-mu
  
  z <- A$P %*% xmu
  
  if (prec) {
    y <- crossprod(A$L,z)  ## L' %*% x
    log.dens <- C + detL - colSums(y*y)/2 
  } else {
    y <- solve(A$L, z) ## Ly = x
    log.dens <- C - detL - colSums(y*y)/2
  }
  
  return(as.numeric(log.dens))
}

