% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/randomizations.R
\name{randomizations}
\alias{randomizations}
\title{Data-stream randomizations}
\usage{
randomizations(
  DT = NULL,
  type = NULL,
  id = NULL,
  group = NULL,
  coords = NULL,
  datetime = NULL,
  splitBy = NULL,
  iterations = NULL
)
}
\arguments{
\item{DT}{input data.table}

\item{type}{one of 'daily', 'step' or 'trajectory' - see details}

\item{id}{Character string of ID column name}

\item{group}{generated from spatial grouping functions - see details}

\item{coords}{Character vector of X coordinate and Y coordinate column names}

\item{datetime}{field used for providing date time or time group - see
details}

\item{splitBy}{List of fields in DT to split the randomization process by}

\item{iterations}{The number of iterations to randomize}
}
\value{
\code{randomizations} returns the random date time or random id along
with the original \code{DT}, depending on the randomization \code{type}.
The length of the returned \code{data.table} is the original number of rows
multiplied by the number of iterations + 1. For example, 3 iterations will
return 4x - one observed and three randomized.

Two columns are always returned: \itemize{ \item observed - if the rows
represent the observed (TRUE/FALSE) \item iteration - iteration of rows
(where 0 is the observed) }

In addition, depending on the randomization type, random ID or random date
time columns are returned:

\itemize{ \item step - \code{randomID} each time step \item daily -
\code{randomID} for each day and \code{jul} indicating julian day \item
trajectory - a random date time ("random" prefixed to \code{datetime}
argument), observed \code{jul} and \code{randomJul} indicating the random
day relocations are swapped to. }
}
\description{
\code{randomizations} performs data-stream social network randomization. The
function accepts a \code{data.table} with relocation data, individual
identifiers and a randomization \code{type}. The \code{data.table} is
randomized either using \code{step} or \code{daily} between-individual
methods, or within-individual daily \code{trajectory} method described by
Spiegel et al. (2016).
}
\details{
The \code{DT} must be a \code{data.table}. If your data is a
\code{data.frame}, you can convert it by reference using
\code{\link[data.table:setDT]{data.table::setDT}}.

Three randomization \code{type}s are provided: \enumerate{ \item step -
randomizes identities of relocations between individuals within each time
step. \item daily - randomizes identities of relocations between individuals
within each day. \item trajectory - randomizes daily trajectories within
individuals (Spiegel et al. 2016). }

Depending on the \code{type}, the \code{datetime} must be a certain format:

\itemize{ \item step - datetime is integer group created by
\code{group_times} \item daily - datetime is \code{POSIXct} format \item
trajectory - datetime is \code{POSIXct} format }

The \code{id}, \code{datetime},  (and optional \code{splitBy}) arguments
expect the names of respective columns in \code{DT} which correspond to the
individual identifier, date time, and additional grouping columns. The
\code{coords} argument is only required when the \code{type} is "trajectory",
since the coordinates are required for recalculating spatial groups with
\code{group_pts}, \code{group_lines} or \code{group_polys}.

Please note that if the data extends over multiple years, a column indicating
the year should be provided to the \code{splitBy} argument. This will ensure
randomizations only occur within each year.

The \code{group} argument is expected only when \code{type} is 'step' or
'daily'.

For example, using \code{\link[data.table:year]{data.table::year}}:

\preformatted{ DT[, yr := year(datetime)] randomizations(DT, type = 'step',
id = 'ID', datetime = 'timegroup', splitBy = 'yr') }

\code{iterations} is set to 1 if not provided. Take caution with a large
value for \code{iterations} with large input \code{DT}.
}
\examples{
# Load data.table
library(data.table)

# Read example data
DT <- fread(system.file("extdata", "DT.csv", package = "spatsoc"))

# Date time columns
DT[, datetime := as.POSIXct(datetime)]
DT[, yr := year(datetime)]

# Temporal grouping
group_times(DT, datetime = 'datetime', threshold = '5 minutes')

# Spatial grouping with timegroup
group_pts(DT, threshold = 5, id = 'ID', coords = c('X', 'Y'), timegroup = 'timegroup')

# Randomization: step
randStep <- randomizations(
    DT,
    type = 'step',
    id = 'ID',
    group = 'group',
    datetime = 'timegroup',
    splitBy = 'yr',
    iterations = 2
)

# Randomization: daily
randDaily <- randomizations(
    DT,
    type = 'daily',
    id = 'ID',
    group = 'group',
    datetime = 'datetime',
    splitBy = 'yr',
    iterations = 2
)

# Randomization: trajectory
randTraj <- randomizations(
    DT,
    type = 'trajectory',
    id = 'ID',
    group = NULL,
    coords = c('X', 'Y'),
    datetime = 'datetime',
    splitBy = 'yr',
    iterations = 2
)

}
\references{
\url{http://onlinelibrary.wiley.com/doi/10.1111/2041-210X.12553/full}
}
\seealso{
Other Social network tools: 
\code{\link{get_gbi}()}
}
\concept{Social network tools}
