% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svmidwpred.R
\name{svmidwpred}
\alias{svmidwpred}
\title{Generate spatial predictions using the hybrid method of support vector machine ('svm')
 regression and inverse distance weighted ('IDW') ('svmidw')}
\usage{
svmidwpred(
  formula = NULL,
  longlat,
  trainxy,
  y,
  longlatpredx,
  predx,
  scale = TRUE,
  type = NULL,
  kernel = "radial",
  degree = 3,
  gamma = if (is.vector(trainxy)) 1 else 1/ncol(trainxy),
  coef0 = 0,
  cost = 1,
  nu = 0.5,
  tolerance = 0.001,
  epsilon = 0.1,
  idp = 2,
  nmaxidw = 12,
  ...
)
}
\arguments{
\item{formula}{a formula defining the response variable and predictive variables
for 'svm'.}

\item{longlat}{a dataframe contains longitude and latitude of point samples. The
location information must be named as 'long' and 'lat'.}

\item{trainxy}{a dataframe contains longitude (long), latitude (lat),
predictive variables and the response variable of point samples.}

\item{y}{a vector of the response variable in the formula, that is, the left
part of the formula.}

\item{longlatpredx}{a dataframe contains longitude and latitude of point locations
(i.e., the centers of grids) to be predicted.}

\item{predx}{a dataframe or matrix contains columns of predictive variables
for the grids to be predicted.}

\item{scale}{A logical vector indicating the variables to be scaled (default: TRUE).}

\item{type}{the default setting is 'NULL'. See '?svm' for various options.}

\item{kernel}{the default setting is 'radial'. See '?svm' for other options.}

\item{degree}{a parameter needed for kernel of type polynomial (default: 3).}

\item{gamma}{a parameter needed for all 'kernels' except 'linear'
(default: 1/(data dimension)).}

\item{coef0}{a parameter needed for kernels of type 'polynomial' and 'sigmoid'(default: 0).}

\item{cost}{cost of constraints violation (default: 1).}

\item{nu}{a parameter needed for 'nu-classification', 'nu-regression', and 'one-classification' (default: 0.5).}

\item{tolerance}{tolerance of termination criterion (default: 0.001).}

\item{epsilon}{'epsilon' in the insensitive-loss function (default: 0.1).
See '?svm' for details.}

\item{idp}{a numeric number specifying the inverse distance weighting power.}

\item{nmaxidw}{for a local predicting: the number of nearest observations that
should be used for a prediction or simulation, where nearest is defined in
terms of the space of the spatial locations. By default, 12 observations
are used.}

\item{...}{other arguments passed on to 'svm'.}
}
\value{
A dataframe of longitude, latitude, and predictions.
}
\description{
This function is for generating spatial predictions using the hybrid
method of 'svm' and 'idw' ('svmidw').
}
\examples{
\donttest{
library(spm)
data(petrel)
data(petrel.grid)

gravel <- petrel[, c(1, 2, 6:9, 5)]
longlat <- petrel[, c(1, 2)]
model <- log(gravel + 1) ~  lat +  bathy + I(long^3) + I(lat^2) + I(lat^3)
y <- log(gravel[, 7] +1)

svmidwpred1 <- svmidwpred(formula = model, longlat = longlat, trainxy =  gravel,
y = y, longlatpredx = petrel.grid[, c(1:2)], predx = petrel.grid, idp = 2,
 nmaxidw = 12)

names(svmidwpred1)

# Back transform 'svmidwpred$predictions' to generate the final predictions
svmidw.predictions <- exp(svmidwpred1$predictions) - 1
range(svmidw.predictions)
}

}
\references{
Li, J., Potter, A., Huang, Z., and Heap, A. (2012). Predicting Seabed
Sand Content across the Australian Margin Using Machine Learning and Geostatistical
 Methods, Geoscience Australia, Record 2012/48, 115pp.

Li, J., Heap, A., Potter, A., and Danilel, J.J. (2011). Predicting Seabed Mud Content
across the Australian Margin II: Performance of Machine Learning Methods and Their
Combination with Ordinary Kriging and Inverse Distance Squared, Geoscience Australia,
Record 2011/07, 69pp.

David Meyer, Evgenia Dimitriadou, Kurt Hornik, Andreas Weingessel and Friedrich
Leisch (2020). e1071: Misc Functions of the Department of Statistics, Probability
Theory Group (Formerly: E1071), TU Wien. R package version 1.7-4.
https://CRAN.R-project.org/package=e1071.

Pebesma, E.J., 2004. Multivariable geostatistics in S: the gstat package.
Computers & Geosciences, 30: 683-691.
}
\author{
Jin Li
}
