#' Calculate the synergy scores for drug combinations
#'
#' A function to calculate the synergy scores for drug combinations using different
#' models.
#'
#' @param data a list object generated by function \code{\link{ReshapeData}}
#' @param method a parameter to specify which models to use to calculate the synergy scores. Choices are "ZIP",
#' "Bliss", "HSA" and "Loewe". Defaults to "ZIP".
#' @param correction a parameter to specify if baseline correction is used or not. Defaults to TRUE.
#' @return a list of the following components:
#' \item{dose.response.mats}{ the same as the input data component.}
#' \item{drug.pairs}{the same as the input data component.}
#' \item{scores}{a list of matrixes of synergy scores.}
#' \item{method}{the method used to calculate the synergy scores.}
#' @author Liye He \email{liye.he@helsinki.fi}
#' @references Yadav B, Wennerberg K, Aittokallio T, Tang J. Searching for Drug Synergy in Complex Dose-Response Landscape Using an Interaction Potency Model.
#' Computational and Structural Biotechnology Journal 2015; 13: 504-513.
#' @examples
#' data("mathews_screening_data")
#' data <- ReshapeData(mathews_screening_data)
#' scores <- CalculateSynergy(data)
CalculateSynergy <- function(data, method = "ZIP", correction = TRUE) {
  if(!is.list(data)) {
    stop("Input data is not a list format!")
  }
  if(!method %in% c("ZIP", "HSA", "Bliss", "Loewe")) {
    stop("The method parameter can only be one of the following: ZIP, HSA, Bliss and Loewe.")
  }
  dose.response.mats <- data$dose.response.mats
  num.pairs <- length(dose.response.mats)
  scores <- list() ## save delta scores for each block
  for (i in 1:num.pairs) {
    response.mat <- dose.response.mats[[i]]
    scores[[i]] <- switch(method,
                          ZIP = ZIP(response.mat, correction),
                          HSA = HSA(response.mat, correction),
                          Bliss = Bliss(response.mat, correction),
                          Loewe = Loewe(response.mat, correction))
  }

  data$scores <- scores
  data$method <- method
  return(data)
}

