% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/draw_table_extra.R
\name{draw_table_extra}
\alias{draw_table_extra}
\title{Graphical display of a table with grobs of varying scales and colours.}
\usage{
draw_table_extra(
  dscale,
  theme,
  output,
  dcolor = NULL,
  dscale_min = NULL,
  dscale_max = NULL,
  cols_more = NULL,
  rows_more = NULL,
  dscale_title_legend = "Scale title",
  dcolor_title_legend = "Color title",
  margin_x = unit(1, "inches"),
  margin_y = unit(1, "inches")
)
}
\arguments{
\item{dscale}{a matrix containing the values defining the grobs scales.}

\item{theme}{a list of theme parameters. Use an instance of \code{ttheme_awesome}.}

\item{output}{path to output file. Only pdf supported for now.}

\item{dcolor}{(optional) a matrix of size (n,m) containing the values defining the grobs colors.}

\item{dscale_min}{(optional) value for setting the minimum scale size of foreground grobs. Entries in the
\code{dscale} matrix below \code{dscale_min} will have a scale of 0 (no grob).}

\item{dscale_max}{(optional) value for setting the maximum scale size of foreground grobs. Entries in the
\code{dscale} matrix above \code{dscale_max} will have a scale of 1.}

\item{cols_more}{(optional) a named list of additional rows (top-part) of the plot for describing the columns The
list names will be used as row headers.}

\item{rows_more}{(optional) a named list of additional columns (right-part) of the plot for describing the rows. The
list names will be used as column headers.}

\item{dscale_title_legend}{(optional) title for the colorbar providing a legend for scales.}

\item{dcolor_title_legend}{(optional) title for the colorbar providing a legend for colors}

\item{margin_x}{(optional) use it to fine-tune the width of the plot if some elements are not displayed correctly.}

\item{margin_y}{(optional) use it to fine-tune the height of the plot if some elements are not displayed correctly.}
}
\value{
No return value, the last instruction calls graphics.off() in order to write the plot to the .pdf file
specified via \code{output} argument.
}
\description{
Draw a table containing grobs of varying sizes and colors encoding two different kinds of information.
The column names and row names of the table are displayed on the top and left sides of the table respectively.
}
\examples{
\donttest{
library(dplyr)
library(tableExtra)
library(tibble)

# load data
load(system.file("testdata", "pcawg_counts.rda", package="tableExtra"))
load(system.file("testdata", "sbs_aetiologies.rda", package="tableExtra"))

pcawg_plot_data <- function(){
  scale_breaks <- seq(from=0, to=1, by=0.1)
  color_palette <- c("#ffc651", "#ffa759", "#ff8962", "#ff6b6b", "#cc6999", "#9968c8", 
                     "#6767f8", "#4459ce", "#224ba5","#013d7c")
  color_breaks <- c(0, 0.05, 0.1, 0.25, 0.5, 1, 2.5, 5, 10, 25, 1e6)
  color_bg <- c("#f8f9fa", "#e9ecef")

  theme <- ttheme_awesome(base_size=12,
                          rep_mode="col",
                          core_size=5, 
                          scale_breaks=scale_breaks,
                          color_palette=color_palette, 
                          color_breaks=color_breaks, 
                          core=list(bg_params=list(fill=color_bg)))

  # define dscale and cols_more from PCAWG data
  dscale <- pcawg_counts \%>\%
    group_by(Cancer.Types) \%>\%
    mutate(n=n()) \%>\%
    summarize_at(vars(-Sample.Names, -Accuracy), ~sum(.x>0)) \%>\%
    mutate_at(vars(-Cancer.Types,-n), ~./n)

  cols_more <- list("n="=dscale$n)
  dscale$n <- NULL
  dscale <- column_to_rownames(.data=dscale, var="Cancer.Types")
  dscale <- t(as.matrix(dscale))
  
  # define dcolor and rows_more from PCAWG data
  mask <- sbs_aetiologies$Signature \%in\% rownames(dscale)
  rows_more <- list("Aetiology"=sbs_aetiologies[mask, "Aetiology"])

  dcolor <- pcawg_counts \%>\%
    group_by(Cancer.Types) \%>\%
    summarize_at(vars(-Sample.Names, -Accuracy), ~median(.[.!=0]*1e6/3.2e9)) \%>\%
    replace(is.na(.),0)

  dcolor <- column_to_rownames(.data=dcolor, var="Cancer.Types")
  dcolor <- t(as.matrix(dcolor))

  list(dscale=dscale, dcolor=dcolor, cols_more=cols_more, rows_more=rows_more, theme=theme)
}

# tables needed for the plot and graphical parameters in `theme`
plot_data <- pcawg_plot_data()

# draw
output <- file.path(tempdir(),"table_extra_pcawg.pdf")
draw_table_extra(dscale=plot_data$dscale, theme=plot_data$theme, output=output,
                 dcolor=plot_data$dcolor, cols_more=plot_data$cols_more,
                 rows_more=plot_data$rows_more,
                 dscale_title_legend="Prop of tumors with the signature",
                 dcolor_title_legend="Median mut/Mb due to signature")
}
}
\seealso{
\code{\link[=ttheme_awesome]{ttheme_awesome()}}, \code{\link[=gtable_table]{gtable_table()}}, \code{\link[=gtable_legend]{gtable_legend()}}
}
\author{
Yoann Pradat
}
