#' Function that creates a factor vector out of a date-time vector.
#'
#' Creates a factor vector out of a date-time vector that an inherits from class "POSIXt" or "Date".
#'
#' @aliases datetime2fac
#' @param p date-time vector
#' @return A factor vector
#' @export
#' @note \code{ff} vectors are not implemented yet
datetime2fac <- function(p) {

	if (!inherits(p, c("Date", "POSIXt"))) stop(paste(parse(substitute(p)), "is not a valid date/time vector"))

	rng <- range(p)


	cutSteps <- c(
		paste(c(50, 20, 10, 5, 4, 2, 1), "year"),
		"quarter",
		"month",
		"week",
		"day",
		paste(c(6, 3, 1), "hour"),
		paste(c(15, 5, 1), "min"),
		paste(c(10, 5, 1), "sec"))
	
	if ("Date" %in% class(p)) {
		cutSteps <- cutSteps[1:11]
	}
	
	lvls <- rep(100, length(cutSteps))
	i <- 1
	for (stp in cutSteps) {
		lvls[i] <- nlevels(cut(rng, breaks=stp))
		if (lvls[i] > 15) break;
		i <- i + 1
	}

	## get last element for which the minimum is closest to 6.4
	idealStpIndex <- length(cutSteps) - which.min(rev(abs(6.4 - lvls))) + 1
	
	if (idealStpIndex %in% c(12,13)) # in case x hours
		startDT <- trunc(rng[1], "days") else
	if (idealStpIndex %in% c(15,16)) # in case x mins
		startDT <- trunc(rng[1], "hours") else
	if (idealStpIndex %in% c(18,19)) # in case x secs
		startDT <- trunc(rng[1], "secs") else startDT <- rng[1]
	p[length(p)+1] <- startDT
	
	p2 <- cut(p, breaks=cutSteps[idealStpIndex])
	p2 <- p2[1:(length(p2)-1)]
	if (idealStpIndex <= 9) levels(p2) <- substr(levels(p2), 1, 7) # shown only years and months
	if (idealStpIndex <= 7) {
		# shown year intervals
		levels(p2) <- substr(levels(p2), 1, 4)
		interval <- (as.numeric(levels(p2)[2]) - as.numeric(levels(p2)[1])) - 1
		levels(p2) <- paste(levels(p2), "-", as.numeric(levels(p2)) + interval, sep = "")
	}

	return(p2)
}