% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/music.R
\name{music}
\alias{music}
\alias{musical}
\alias{as_music}
\alias{is_music}
\alias{music_split}
\title{Create music objects and check music string validity}
\usage{
musical(x)

as_music(
  notes,
  info = NULL,
  lyrics = NA,
  key = "c",
  time = "4/4",
  tempo = "2 = 60",
  accidentals = NULL,
  format = NULL,
  labels = NULL,
  at = seq_along(labels)
)

is_music(x)

music_split(x)
}
\arguments{
\item{x}{character or music, a string to be coerced or an existing music
object.}

\item{notes, info}{noteworthy and note info strings. For \code{as_music()}, a
complete music string is assumed for \code{notes} when \code{info = NULL}.}

\item{lyrics}{optional \code{lyrics} object or \code{NA}, attached to output as an
attribute.}

\item{key}{character, store the key signature as a music attribute. Defaults
to \code{"c"}. See details.}

\item{time}{character, store the time signature as a music attribute.
Defaults to \code{"4/4"}. See details.}

\item{tempo}{character or \code{NA}, defaults to \code{"2 = 60"}. See details.}

\item{accidentals}{\code{NULL} or character, represent accidentals, \code{"flat"} or
\code{"sharp"}.}

\item{format}{\code{NULL} or character, the timestep delimiter format, \code{"space"}
or \code{"vector"}.}

\item{labels}{character, text annotations to attach to timesteps using
\code{notate}.}

\item{at}{integer, timesteps for \code{labels}, defaults to starting from
time one.}
}
\value{
depends on the function
}
\description{
Check whether a string is comprised exclusively of valid syntax for music
strings. A music object can be built from such a string. It combines a
noteworthy string and a note info string.
}
\details{
With note info strings, you are required to enter something at every
timestep, even if it is only the duration. This makes sense because if you
do not enter something, there is simply no indication of a timestep.
A nice feature of music strings is that explicit timesteps are achieved
just by having notes present, allowing you to leave out durations entirely
when they repeat, inheriting them from the previous timestep where duration
was given explicitly. There is no need to enter the same number across
consecutive timesteps; the first will suffice and the rest are automatically
filled in for you when the object is constructed.

\code{musical()} returns a scalar logical result indicating whether all timesteps
contain exclusively valid entries.

\code{as_music()} can be used to coerce to the \code{music} class. Coercion will fail
if the string is not musical. The \code{music} class has its own \code{print()} and
\code{summary()} methods. \code{music} objects are primarily intended to represent an
aggregation of a \code{noteworthy} object and a \code{noteinfo}. You can optionally
fold in a \code{lyrics} object as well. However, for music data analysis, any
operations will involve first splitting the object into its component parts.
The value of this class is for the more efficient data entry it provides.

When \code{accidentals} or \code{format} are \code{NULL}, these settings are inferred from
the musical string input. When mixed formats are present, flats are the
default for accidentals.

Other attributes are attached to a \code{music} object. \code{key} uses the \code{tabr}
syntax, e.g., \code{"c"}, \code{"b_"}, \code{"f#m"}, etc. \code{time} and \code{tempo} use the
LilyPond string format. For music programming and analysis, \code{key}, \code{time} and
\code{tempo} can most likely be ignored. They are primarily relevant when
rendering a music snippet directly from a \code{music} object with LilyPond.
These additional attributes provide more complete context for the rendered
sheet music.

If you plan to render music snippets from a \code{music} object that you are
defining from a new character string, and the context you have in mind is a
stringed and fretted instrument like guitar, you can specify string numbers
at the end of each timestep with numbers following a semicolon delimiter.
This would still precede any \code{*} timestep multiplier number. See examples.

Note that if you convert a music object to a phrase object, you are changing
contexts. The phrase object is the simplest LilyPond-format music structure.
Coercion with \code{phrase()} strips all attributes of a music object and
retains only notes, note info and string numbers.
}
\examples{
# note durations inherit from previous timestep if missing
x <- "a#4-+ b_[staccato] c,x d''t8( e)( g_')- a4 c,e_,g, ce_g4. a~8 a1"
is_music(x)
musical(x)
x <- as_music(x)
is_music(x)
x

y <- lyrics_template(x)
y[3:8] <- strsplit("These are some song ly- rics", " ")[[1]]
y

x <- as_music(x, lyrics = y, accidentals = "sharp")
summary(x)

# Starting string = 5: use ';5'. Carries over until an explicit change.
x <- "a,4;5*5 b,4-+ c4[staccato] cgc'e'~4 cgc'e'1 e'4;2 c';3 g;4 c;5 ce'1;51"
x <- as_music_df(as_music(x))
x$string
}
\seealso{
\code{\link[=music-helpers]{music-helpers()}}, \code{\link[=note-checks]{note-checks()}}, \code{\link[=note-metadata]{note-metadata()}},
\code{\link[=note-summaries]{note-summaries()}}, \code{\link[=note-coerce]{note-coerce()}}
}
