#' @title A `drake`-plan-like pipeline archetype
#' @description Simplify target specification in pipelines.
#' @details Allows targets with just targets and commands
#'   to be written in the pipeline as `target = command` instead of
#'   `tar_target(target, command)`. Also supports ordinary
#'   target objects if they are unnamed.
#'   `tar_plan(x = 1, y = 2, tar_target(z, 3), tar_render(r, "r.Rmd"))`
#'   is equivalent to
#'   `list(tar_target(x, 1), tar_target(y, 2), tar_target(z, 3), tar_render(r, "r.Rmd"))`. # nolint
#' @export
#' @return A list of `tar_target()` objects.
#'   Target objects represent skippable steps of the analysis pipeline
#'   as described at <https://books.ropensci.org/targets/>.
#'   Please see the design specification at
#'   <https://books.ropensci.org/targets-design/>
#'   to learn about the structure and composition of target objects.
#' @param ... Named and unnamed targets. All named targets must follow
#'   the `drake`-plan-like `target = command` syntax, and all unnamed
#'   arguments must be explicit calls to create target objects,
#'   e.g. `tar_target()`, target archetypes like [tar_render()], or similar.
#' @examples
#' if (identical(Sys.getenv("TAR_LONG_EXAMPLES"), "true")) {
#' targets::tar_dir({ # tar_dir() runs code from a temporary directory.
#' targets::tar_script({
#'   library(tarchetypes)
#'   tar_plan(
#'     tarchetypes::tar_fst_tbl(data, data.frame(x = seq_len(26))),
#'     means = colMeans(data) # No need for tar_target() for simple cases.
#'   )
#' })
#' targets::tar_make()
#' })
#' }
tar_plan <- function(...) {
  commands <- tar_plan_parse(match.call(expand.dots = FALSE)$...)
  lapply(commands, eval, envir = targets::tar_option_get("envir"))
}

tar_plan_parse <- function(commands) {
  names <- names(commands) %|||% rep("", length(commands))
  is_named <- !is.na(names) & nzchar(names)
  commands[is_named] <- tar_plan_parse_named(commands[is_named])
  commands
}

tar_plan_parse_named <- function(commands) {
  lapply(names(commands), tar_plan_parse_command, commands = commands)
}

tar_plan_parse_command <- function(name, commands) {
  env <- list(name = as.symbol(name), command = commands[[name]])
  substitute(targets::tar_target(name, command), env = env)
}
