% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/package.R
\docType{package}
\name{taxa-package}
\alias{taxa-package}
\title{taxa}
\description{
The \code{taxa} package is intended to:
\itemize{
\item Provide a set of classes to store taxonomic data and any user-specific data associated with it
\item Provide functions to convert commonly used formats to these classes
\item Provide a common foundation for other packages to build on to enable an ecosystem of compatible packages dealing with taxonomic data.
\item Provide generally useful functionality, such as filtering and mapping functions
}
}
\section{Main classes}{


These are the classes users would typically interact with:
\itemize{
\item \link{taxon}: A class used to define a single taxon. Many other classes in the
`taxa`` package include one or more objects of this class.
\item \link{taxa}: Stores one or more \link{taxon} objects. This is just a thin wrapper
for a list of \link{taxon} objects.
\item \link{hierarchy}: A class containing an ordered list of \link{taxon} objects that
represent a hierarchical classification.
\item \link{hierarchies}: A list of taxonomic classifications.  This is just a thin wrapper
for a list of \link{hierarchy} objects.
\item \link{taxonomy}: A taxonomy composed of \link{taxon} objects organized in a tree
structure. This differs from the \link{hierarchies} class in how the \link{taxon}
objects are stored. Unlike a \link{hierarchies} object, each unique taxon is
stored only once and the relationships between taxa are stored in an
edgelist.
\item \link{taxmap}: A class designed to store a taxonomy and associated
user-defined data. This class builds on the \link{taxonomy} class. User defined
data can be stored in the list \code{obj$data}, where \code{obj} is a taxmap
object. Any number of user-defined lists, vectors, or tables mapped
to taxa can be manipulated in a cohesive way such that relationships
between taxa and data are preserved.
}
}

\section{Minor classes}{


These classes are mostly components for the larger classes above and would
not typically be used on their own.
\itemize{
\item \link{taxon_database}: Used to store information about taxonomy databases.
\item \link{taxon_id}: Used to store taxon IDs, either arbitrary or from a
particular taxonomy database.
\item \link{taxon_name}: Used to store taxon names, either arbitrary or from a
particular taxonomy database.
\item \link{taxon_rank}: Used to store taxon ranks (e.g. species, family), either
arbitrary or from a particular taxonomy database.
}
}

\section{Major manipulation functions}{


These are some of the more important functions used to filter data in classes
that store multiple taxa, like \link{hierarchies}, \link{taxmap}, and \link{taxonomy}.
\itemize{
\item \link{filter_taxa}: Filter taxa in a \link{taxonomy} or \link{taxmap} object with a
series of conditions. Relationships between remaining taxa and user-defined
data are preserved (There are many options controlling this).
\item \link{filter_obs}: Filter user-defined data \link{taxmap} object with a series of
conditions. Relationships between remaining taxa and user-defined data are
preserved (There are many options controlling this);
\item \link{sample_n_taxa}: Randomly sample taxa. Has same abilities as
\link{filter_taxa}.
\item \link{sample_n_obs}: Randomly sample observations. Has same abilities as
\link{filter_obs}.
\item \link{mutate_obs}: Add datasets or columns to datasets in the \code{data} list of
\link{taxmap} objects.
\item \link{pick}: Pick out specific taxa, while others are dropped in \link{hierarchy}
and \link{hierarchies} objects.
\item \link{pop}: Pop out taxa (drop them) in \link{hierarchy} and \link{hierarchies} objects.
\item \link{span}: Select a range of taxa, either by two names, or relational
operators in \link{hierarchy} and \link{hierarchies} objects.
}
}

\section{Mapping functions}{


There are lots of functions for getting information for each taxon.
\itemize{
\item \link{subtaxa}: Return data for the subtaxa of each taxon in an \link{taxonomy} or
\link{taxmap} object.
\item \link{supertaxa}: Return data for the supertaxa of each taxon in an \link{taxonomy}
or \link{taxmap} object.
\item \link{roots}: Return data for the roots of each taxon in an \link{taxonomy} or
\link{taxmap} object.
\item \link{leaves}: Return data for the leaves of each taxon in an \link{taxonomy} or
\link{taxmap} object.
\item \link{obs}: Return user-specific data for each taxon and all of its subtaxa in
an \link{taxonomy} or \link{taxmap} object.
}
}

\section{The kind of classes used}{


Note, this is mostly of interest to developers and advanced users.

The classes in the \code{taxa} package are mostly
\href{https://adv-r.hadley.nz/r6.html}{R6} classes (\link{R6Class}). A few of the
simpler ones (\link{taxa} and \link{hierarchies}) are
\href{https://adv-r.hadley.nz/s3.html}{S3} instead. R6 classes are different than
most R objects because they are
\href{https://en.wikipedia.org/wiki/Immutable_object}{mutable} (e.g. A function
can change its input without returning it). In this, they are more similar
to class systems in
\href{https://en.wikipedia.org/wiki/Object-oriented_programming}{object-oriented}
languages like python. As in other object-oriented class systems, functions
are thought to "belong" to classes (i.e. the data), rather than functions
existing independently of the data. For example, the function \code{print} in R
exists apart from what it is printing, although it will change how it prints
based on what the class of the data is that is passed to it. In fact, a user
can make a custom print method for their own class by defining a function
called \code{print.myclassname}. In contrast, the functions that operate on R6
functions are "packaged" with the data they operate on. For example, a print
method of an object for an R6 class might be called like
\code{my_data$print()} instead of \code{print(my_data)}.
}

\section{The two ways to call functions}{


Note, you will need to read the previous section to fully understand this one.

Since the R6 function syntax (e.g. \code{my_data$print()}) might be confusing to
many R users, all functions in \code{taxa} also have S3 versions. For example,
the \code{\link[=filter_taxa]{filter_taxa()}} function can be called on a \link{taxmap} object called
\code{my_obj} like \code{my_obj$filter_taxa(...)} (the R6 syntax) or
\code{filter_taxa(my_obj, ...)} (the S3 syntax). For some functions, these two
way of calling the function can have different effect. For functions that do
not returned a modified version of the input (e.g. \code{\link[=subtaxa]{subtaxa()}}), the two ways have identical behavior.
However, functions like \code{\link[=filter_taxa]{filter_taxa()}}, that modify their inputs, actually
change the object passed to them as the first argument as well as returning that
object. For example,

\code{my_obj <- filter_taxa(my_obj, ...)}

and

\code{my_obj$filter_taxa(...)}

and

\code{new_obj <- my_obj$filter_taxa(...)}

all replace \code{my_obj} with the filtered result, but

\code{new_obj <- filter_taxa(my_obj, ...)}

will not modify \code{my_obj}.
}

\section{Non-standard evaluation}{


This is a rather advanced topic.

Like packages such as \code{ggplot2} and \link{dplyr}, the \code{taxa} package uses
non-standard evaluation to allow code
to be more readable and shorter. In effect, there are variables that only
"exist" inside a function call and depend on what is passed to that function
as the first parameter (usually a class object). For example, in the \code{dpylr}
function \code{\link[=filter]{filter()}}, column names can be used as if they were independent
variables. See \code{?dpylr::filter} for examples of this. The \code{taxa} package builds on this idea.

For many functions that work on \link{taxonomy} or \link{taxmap} objects (e.g. \link{filter_taxa}),
some functions that return per-taxon information (e.g. \code{\link[=taxon_names]{taxon_names()}}) can
be referred to by just the name of the function. When one of these functions
are referred to by name, the function is run on the relevant object and its
value replaces the function name. For example,

\code{new_obj <- filter_taxa(my_obj, taxon_names == "Bacteria")}

is identical to:

\code{new_obj <- filter_taxa(my_obj, taxon_names(my_obj) == "Bacteria")}

which is identical to:

\code{new_obj <- filter_taxa(my_obj, my_obj$taxon_names() == "Bacteria")}

which is identical to:

\code{my_names <- taxon_names(my_obj)}

\code{new_obj <- filter_taxa(my_obj, my_names == "Bacteria")}

For \code{taxmap} objects, you can also use names of user defined lists, vectors,
and the names of columns in user-defined tables that are stored in the
\code{obj$data} list. See \code{\link[=filter_taxa]{filter_taxa()}} for examples. You can even add your own
functions that are called by name by adding them to the \code{obj$funcs} list.
For any object with functions that use non-standard evaluation, you can see
what values can be used  with \code{\link[=all_names]{all_names()}} like \code{all_names(obj)}.
}

\section{Dependencies and inspiration}{


Various elements of the \code{taxa} package were inspired by the \link{dplyr} and
\link{taxize} packages. This package started as parts of the \code{metacoder} and
\code{binomen} packages. There are also many dependencies that make \code{taxa}
possible.
}

\section{Feedback and contributions}{


Find a problem? Have a suggestion? Have a question? Please submit an issue
at our \href{https://github.com/ropensci/taxa}{GitHub repository}:

\url{https://github.com/ropensci/taxa/issues}

A GitHub account is free and easy to set up. We welcome feedback! If you
don't want to use GitHub for some reason, feel free to email us. We do
prefer posting to github since it allows others that might have the same
issue to see our conversation. It also helps us keep track of what problems
we need to address.

Want to contribute code or make a change to the code? Great, thank you!
Please \href{https://help.github.com/articles/fork-a-repo/}{fork} our GitHub
repository and submit a \href{https://help.github.com/articles/about-pull-requests/}{pull request}.
}

\section{For more information}{


Checkout the vignette
(\code{browseVignettes("taxa")}) for detailed introduction and examples.
}

\author{
Scott Chamberlain \email{myrmecocystus+r@gmail.com}

Zachary Foster \email{zacharyfoster1989@gmail.com}
}
\keyword{package}
