% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prop_diff.R
\name{h_prop_diff}
\alias{h_prop_diff}
\alias{prop_diff_wald}
\alias{prop_diff_ha}
\alias{prop_diff_nc}
\alias{prop_diff_cmh}
\alias{prop_diff_strat_nc}
\title{Helper functions to calculate proportion difference}
\usage{
prop_diff_wald(rsp, grp, conf_level = 0.95, correct = FALSE)

prop_diff_ha(rsp, grp, conf_level)

prop_diff_nc(rsp, grp, conf_level, correct = FALSE)

prop_diff_cmh(rsp, grp, strata, conf_level = 0.95)

prop_diff_strat_nc(
  rsp,
  grp,
  strata,
  weights_method = c("cmh", "wilson_h"),
  conf_level = 0.95,
  correct = FALSE
)
}
\arguments{
\item{rsp}{(\code{logical})\cr vector indicating whether each subject is a responder or not.}

\item{grp}{(\code{factor})\cr vector assigning observations to one out of two groups
(e.g. reference and treatment group).}

\item{conf_level}{(\code{proportion})\cr confidence level of the interval.}

\item{correct}{(\code{flag})\cr whether to include the continuity correction. For further
information, see \code{\link[stats:prop.test]{stats::prop.test()}}.}

\item{strata}{(\code{factor})\cr variable with one level per stratum and same length as \code{rsp}.}

\item{weights_method}{(\code{string})\cr weights method. Can be either \code{"cmh"} or \code{"heuristic"}
and directs the way weights are estimated.}
}
\value{
A named \code{list} of elements \code{diff} (proportion difference) and \code{diff_ci}
(proportion difference confidence interval).
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}
}
\section{Functions}{
\itemize{
\item \code{prop_diff_wald()}: The Wald interval follows the usual textbook
definition for a single proportion confidence interval using the normal
approximation. It is possible to include a continuity correction for Wald's
interval.

\item \code{prop_diff_ha()}: Anderson-Hauck confidence interval.

\item \code{prop_diff_nc()}: Newcombe confidence interval. It is based on
the Wilson score confidence interval for a single binomial proportion.

\item \code{prop_diff_cmh()}: Calculates the weighted difference. This is defined as the difference in
response rates between the experimental treatment group and the control treatment group, adjusted
for stratification factors by applying Cochran-Mantel-Haenszel (CMH) weights. For the CMH chi-squared
test, use \code{\link[stats:mantelhaen.test]{stats::mantelhaen.test()}}.

\item \code{prop_diff_strat_nc()}: Calculates the stratified Newcombe confidence interval and difference in response
rates between the experimental treatment group and the control treatment group, adjusted for stratification
factors. This implementation follows closely the one proposed by \insertCite{Yan2010-jt;textual}{tern}.
Weights can be estimated from the heuristic proposed in \code{\link[=prop_strat_wilson]{prop_strat_wilson()}} or from CMH-derived weights
(see \code{\link[=prop_diff_cmh]{prop_diff_cmh()}}).

}}
\examples{
# Wald confidence interval
set.seed(2)
rsp <- sample(c(TRUE, FALSE), replace = TRUE, size = 20)
grp <- factor(c(rep("A", 10), rep("B", 10)))

prop_diff_wald(rsp = rsp, grp = grp, conf_level = 0.95, correct = FALSE)

# Anderson-Hauck confidence interval
## "Mid" case: 3/4 respond in group A, 1/2 respond in group B.
rsp <- c(TRUE, FALSE, FALSE, TRUE, TRUE, TRUE)
grp <- factor(c("A", "B", "A", "B", "A", "A"), levels = c("B", "A"))

prop_diff_ha(rsp = rsp, grp = grp, conf_level = 0.90)

## Edge case: Same proportion of response in A and B.
rsp <- c(TRUE, FALSE, TRUE, FALSE)
grp <- factor(c("A", "A", "B", "B"), levels = c("A", "B"))

prop_diff_ha(rsp = rsp, grp = grp, conf_level = 0.6)

# Newcombe confidence interval

set.seed(1)
rsp <- c(
  sample(c(TRUE, FALSE), size = 40, prob = c(3 / 4, 1 / 4), replace = TRUE),
  sample(c(TRUE, FALSE), size = 40, prob = c(1 / 2, 1 / 2), replace = TRUE)
)
grp <- factor(rep(c("A", "B"), each = 40), levels = c("B", "A"))
table(rsp, grp)

prop_diff_nc(rsp = rsp, grp = grp, conf_level = 0.9)

# Cochran-Mantel-Haenszel confidence interval

set.seed(2)
rsp <- sample(c(TRUE, FALSE), 100, TRUE)
grp <- sample(c("Placebo", "Treatment"), 100, TRUE)
grp <- factor(grp, levels = c("Placebo", "Treatment"))
strata_data <- data.frame(
  "f1" = sample(c("a", "b"), 100, TRUE),
  "f2" = sample(c("x", "y", "z"), 100, TRUE),
  stringsAsFactors = TRUE
)

prop_diff_cmh(
  rsp = rsp, grp = grp, strata = interaction(strata_data),
  conf_level = 0.90
)

# Stratified Newcombe confidence interval

set.seed(2)
data_set <- data.frame(
  "rsp" = sample(c(TRUE, FALSE), 100, TRUE),
  "f1" = sample(c("a", "b"), 100, TRUE),
  "f2" = sample(c("x", "y", "z"), 100, TRUE),
  "grp" = sample(c("Placebo", "Treatment"), 100, TRUE),
  stringsAsFactors = TRUE
)

prop_diff_strat_nc(
  rsp = data_set$rsp, grp = data_set$grp, strata = interaction(data_set[2:3]),
  weights_method = "cmh",
  conf_level = 0.90
)

prop_diff_strat_nc(
  rsp = data_set$rsp, grp = data_set$grp, strata = interaction(data_set[2:3]),
  weights_method = "wilson_h",
  conf_level = 0.90
)

}
\references{
\insertRef{Yan2010-jt}{tern}
}
\seealso{
\code{\link[=prop_diff]{prop_diff()}} for implementation of these helper functions.
}
