% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/excel-if-functions.R
\name{excel_if_functions}
\alias{excel_if_functions}
\alias{SUM_IFS}
\alias{COUNT_IFS}
\alias{AVERAGE_IFS}
\alias{MEDIAN_IFS}
\alias{MIN_IFS}
\alias{MAX_IFS}
\alias{CREATE_IFS}
\title{Excel Summarising "If" Functions}
\usage{
SUM_IFS(x, ...)

COUNT_IFS(x, ...)

AVERAGE_IFS(x, ...)

MEDIAN_IFS(x, ...)

MIN_IFS(x, ...)

MAX_IFS(x, ...)

CREATE_IFS(.f, ...)
}
\arguments{
\item{x}{A vector. Most functions are designed for numeric data.
Some functions like \code{\link[=COUNT_IFS]{COUNT_IFS()}} handle multiple data types.}

\item{...}{Add cases to evaluate. See Details.}

\item{.f}{A function to convert to an "IFS" function.
Use \code{...} in this case to provide parameters to the \code{.f} like \code{na.rm = TRUE}.}
}
\value{
\itemize{
\item \strong{Summary functions} return a single value
}
}
\description{
\strong{"IFS" functions} are filtering versions of their summarization counterparts.
Simply add "cases" that filter if a condition is true.
Multiple cases are evaluated as "AND" filtering operations.
A single case with \code{|} ("OR") bars can be created to accomplish an "OR".
See details below.

These functions are designed to help users coming from an \strong{Excel background}.
Most functions replicate the behavior of Excel:
\itemize{
\item Names are similar to Excel function names
\item By default, missing values are ignored (same as in Excel)
}
}
\details{
\strong{"AND" Filtering:}
Multiple cases are evaluated as "AND" filtering operations.

\strong{"OR" Filtering:}
Compound single cases with \code{|} ("OR") bars can be created to accomplish an "OR".
Simply use a statement like \code{x > 10 | x < -10} to perform an "OR" if-statement.

\strong{Creating New "Summarizing IFS" Functions:}
Users can create new "IFS" functions using the \code{\link[=CREATE_IFS]{CREATE_IFS()}} function factory.
The only requirement is that the output of your function (\code{.f}) must be a single
value (scalar). See examples below.
}
\section{Useful Functions}{


\strong{Summary Functions} - Return a single value from a vector
\itemize{
\item Sum: \code{\link[=SUM_IFS]{SUM_IFS()}}
\item Center: \code{\link[=AVERAGE_IFS]{AVERAGE_IFS()}}, \code{\link[=MEDIAN_IFS]{MEDIAN_IFS()}}
\item Count: \code{\link[=COUNT_IFS]{COUNT_IFS()}}
\item Range: \code{\link[=MIN_IFS]{MIN_IFS()}}, \code{\link[=MAX_IFS]{MAX_IFS()}}
}

\strong{Create your own summary "IFS" function}
\itemize{
\item \code{\link[=CREATE_IFS]{CREATE_IFS()}}: This is a function factory that generates summary "_IFS" functions.
}
}

\examples{
library(tidyverse)
library(tidyquant)
library(timetk)
library(stringr)
library(lubridate)

# --- Basic Usage ---

SUM_IFS(x = 1:10, x > 5)

COUNT_IFS(x = letters, str_detect(x, "a|b|c"))

SUM_IFS(-10:10, x > 8 | x < -5)

# Create your own IFS function (Mind blowingly simple)!
Q75_IFS <- CREATE_IFS(.f = quantile, probs = 0.75, na.rm = TRUE)
Q75_IFS(1:10, x > 5)

# --- Usage with tidyverse ---

# Using multiple cases IFS cases to count the frequency of days with
# high trade volume in a given year
FANG \%>\%
    group_by(symbol) \%>\%
    summarise(
        high_volume_in_2015 = COUNT_IFS(volume,
                                        year(date) == 2015,
                                        volume > quantile(volume, 0.75))
    )

# Count negative returns by month
FANG \%>\%
    mutate(symbol = as_factor(symbol)) \%>\%
    group_by(symbol) \%>\%

    # Collapse from daily to FIRST value by month
    summarise_by_time(
        .date_var  = date,
        .by        = "month",
        adjusted   = FIRST(adjusted)
    ) \%>\%

    # Calculate monthly returns
    group_by(symbol) \%>\%
    mutate(
        returns = PCT_CHANGE(adjusted, fill_na = 0)
    ) \%>\%

    # Find returns less than zero and count the frequency
    summarise(
        negative_monthly_returns = COUNT_IFS(returns, returns < 0)
    )

}
