% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mallet_tidiers.R
\name{mallet_tidiers}
\alias{mallet_tidiers}
\alias{tidy.jobjRef}
\alias{augment.jobjRef}
\title{Tidiers for Latent Dirichlet Allocation models from the mallet package}
\usage{
\method{tidy}{jobjRef}(
  x,
  matrix = c("beta", "gamma"),
  log = FALSE,
  normalized = TRUE,
  smoothed = TRUE,
  ...
)

\method{augment}{jobjRef}(x, data, ...)
}
\arguments{
\item{x}{A jobjRef object, of type RTopicModel, such as created
by \code{\link[mallet]{MalletLDA}}.}

\item{matrix}{Whether to tidy the beta (per-term-per-topic, default)
or gamma (per-document-per-topic) matrix.}

\item{log}{Whether beta/gamma should be on a log scale, default FALSE}

\item{normalized}{If true (default), normalize so that each
document or word sums to one across the topics. If false, values will
be integers representing the actual number of word-topic or document-topic
assignments.}

\item{smoothed}{If true (default), add the smoothing parameter to each
to avoid any values being zero. This smoothing parameter is initialized
as \code{alpha.sum} in \code{\link[mallet]{MalletLDA}}.}

\item{...}{Extra arguments, not used}

\item{data}{For \code{augment}, the data given to the LDA function, either
as a DocumentTermMatrix or as a tidied table with "document" and "term"
columns.}
}
\value{
\code{augment} must be provided a data argument containing
one row per original document-term pair, such as is returned by
\link{tdm_tidiers}, containing columns \code{document} and \code{term}.
It returns that same data with an additional column
\code{.topic} with the topic assignment for that document-term combination.
}
\description{
Tidy LDA models fit by the mallet package, which wraps the Mallet topic
modeling package in Java. The arguments and return values
are similar to \code{\link{lda_tidiers}}.
}
\details{
Note that the LDA models from \code{\link[mallet]{MalletLDA}}
are technically a special case of S4 objects with class \code{jobjRef}.
These are thus implemented as \code{jobjRef} tidiers, with a check for
whether the \code{toString} output is as expected.
}
\examples{

\dontrun{
library(mallet)
library(dplyr)

data("AssociatedPress", package = "topicmodels")
td <- tidy(AssociatedPress)

# mallet needs a file with stop words
tmp <- tempfile()
writeLines(stop_words$word, tmp)

# two vectors: one with document IDs, one with text
docs <- td \%>\%
  group_by(document = as.character(document)) \%>\%
  summarize(text = paste(rep(term, count), collapse = " "))

docs <- mallet.import(docs$document, docs$text, tmp)

# create and run a topic model
topic_model <- MalletLDA(num.topics = 4)
topic_model$loadDocuments(docs)
topic_model$train(20)

# tidy the word-topic combinations
td_beta <- tidy(topic_model)
td_beta

# Examine the four topics
td_beta \%>\%
  group_by(topic) \%>\%
  top_n(8, beta) \%>\%
  ungroup() \%>\%
  mutate(term = reorder(term, beta)) \%>\%
  ggplot(aes(term, beta)) +
  geom_col() +
  facet_wrap(~ topic, scales = "free") +
  coord_flip()

# find the assignments of each word in each document
assignments <- augment(topic_model, td)
assignments
}

}
\seealso{
\code{\link{lda_tidiers}}, \code{\link[mallet]{mallet.doc.topics}},
\code{\link[mallet]{mallet.topic.words}}
}
