## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 7, fig.height = 4,
  dpi = 96,
  message = FALSE, warning = FALSE
)
# Guard to avoid interactive widgets on CRAN
eval_interactive <- interactive()

## -----------------------------------------------------------------------------
library(tikatuwq)

# Try to make wq_demo available even when the package
# cannot be temporarily reinstalled during pkgdown build.
ok <- FALSE
try({
  utils::data("wq_demo", package = "tikatuwq", envir = environment())
  ok <- exists("wq_demo", inherits = FALSE)
}, silent = TRUE)

if (!ok) {
  # Fallback: read example CSV shipped with the package
  csv <- system.file("extdata", "exemplo_chamagunga.csv", package = "tikatuwq")
  if (nzchar(csv)) {
    wq_demo <- read_wq(csv)
    ok <- TRUE
  }
}

if (!ok) {
  # Last resort: small synthetic dataset (schema-compatible)
  set.seed(1)
  wq_demo <- data.frame(
    data = as.Date("2025-01-01") + 0:9,
    rio = "Demo",
    ponto = paste0("P", seq_len(10)),
    turbidez = runif(10, 1, 50),
    od = runif(10, 5, 9),
    pH = runif(10, 6.5, 8.5),
    temperatura = runif(10, 20, 28),
    condutividade = runif(10, 50, 300),
    stringsAsFactors = FALSE
  )
}

# Coerce common numeric columns defensively (some example files may carry strings)
num_candidates <- c(
  "turbidez","od","pH","temperatura","condutividade",
  "dbo","dbo5","nitrato","fosforo","amonia",
  "coliformes","coliformes_totais","coliformes_termotolerantes"
)
for (nm in intersect(num_candidates, names(wq_demo))) {
  suppressWarnings({ wq_demo[[nm]] <- as.numeric(wq_demo[[nm]]) })
}

head(wq_demo)

## -----------------------------------------------------------------------------
ok_iqa <- TRUE
df <- tryCatch({
  wq_demo |>
    validate_wq() |>
    iqa(na_rm = TRUE)
}, error = function(e) {
  ok_iqa <<- FALSE
  message("iqa() failed in vignette build: ", conditionMessage(e))
  wq_demo |>
    validate_wq()
})

conf <- tryCatch({
  conama_check(df, classe = "2")
}, error = function(e) {
  message("conama_check() failed in vignette build: ", conditionMessage(e))
  NULL
})

if (!is.null(conf)) head(conf)

## -----------------------------------------------------------------------------
if (isTRUE(ok_iqa)) {
  plot_iqa(df)
}

## -----------------------------------------------------------------------------
conf_long <- tryCatch(conama_summary(df, classe = "2"), error = function(e) NULL)
if (!is.null(conf_long)) head(conf_long)

## ----eval=FALSE---------------------------------------------------------------
# # Avoid running nested rendering during vignette build on CRAN
# out_file <- render_report(df)
# out_file

## -----------------------------------------------------------------------------
if (eval_interactive) {
  # Minimal example dataset with coordinates
  df_map <- data.frame(
    rio = c("Buranhem","Chamagunga"),
    ponto = c("P1","P2"),
    data = as.Date(c("2025-09-20","2025-09-21")),
    latitude  = c(-16.435, -16.498),
    longitude = c(-39.062, -39.080),
    iqa = c(72, 58)
  )

  plot_map(
    df_map,
    popup_cols = c("rio","ponto","data","iqa"),
    color_by = "iqa"
  )
}

## -----------------------------------------------------------------------------
if (eval_interactive) {
  plot_map(
    df_map,
    popup_cols = c("rio","ponto","data","iqa"),
    cluster = FALSE,
    tiles = "CartoDB.Positron"
  )
}

