% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{tinyVAST}
\alias{tinyVAST}
\title{Fit vector autoregressive spatio-temporal model}
\usage{
tinyVAST(
  formula,
  data,
  time_term = NULL,
  space_term = NULL,
  spacetime_term = NULL,
  family = gaussian(),
  space_columns = c("x", "y"),
  spatial_domain = NULL,
  time_column = "time",
  times = NULL,
  variable_column = "var",
  variables = NULL,
  distribution_column = "dist",
  delta_options = list(formula = ~1),
  spatial_varying = NULL,
  weights = NULL,
  control = tinyVASTcontrol()
)
}
\arguments{
\item{formula}{Formula with response on left-hand-side and predictors on right-hand-side,
parsed by \code{mgcv} and hence allowing \code{s(.)} for splines or \code{offset(.)} for
an offset.}

\item{data}{Data-frame of predictor, response, and offset variables.  Also includes
variables that specify space, time, variables, and the distribution for samples,
as identified by arguments \code{variable_column}, \code{time_column}, \code{space_columns},
and \code{distribution_column}.}

\item{time_term}{Specification for time-series structural equation model structure for
constructing a time-variable interaction that defines a time-varying intercept
for each variable (i.e., applies uniformly across space).
\code{time_term=NULL} disables the space-variable interaction;
see \code{\link[=make_dsem_ram]{make_dsem_ram()}} for notation.}

\item{space_term}{Specification for structural equation model structure for
constructing a space-variable interaction.
\code{space_term=NULL} disables the space-variable interaction;
see \code{\link[=make_sem_ram]{make_sem_ram()}} for notation.}

\item{spacetime_term}{Specification for time-series structural equation model structure
including lagged or simultaneous effects for
constructing a time-variable interaction, which is then combined in
a separable process with the spatial correlation to form a
space-time-variable interaction (i.e., the interaction occurs locally at each site).
\code{spacetime_term=NULL} disables the space-variable interaction; see
\code{\link[=make_dsem_ram]{make_dsem_ram()}}  or \code{\link[=make_eof_ram]{make_eof_ram()}}.}

\item{family}{A function returning a class \code{family}, including \code{\link[=gaussian]{gaussian()}},
\code{\link[=lognormal]{lognormal()}}, \code{\link[=tweedie]{tweedie()}},  \code{\link[=binomial]{binomial()}},  \code{\link[=Gamma]{Gamma()}}, \code{\link[=poisson]{poisson()}},
\code{\link[=nbinom1]{nbinom1()}}, or \code{\link[=nbinom2]{nbinom2()}}.
Alternatively, can be a named list of
these functions, with names that match levels of
\code{data$distribution_column} to allow different
families by row of data. Delta model families are possible, and see
\code{\link[tinyVAST:families]{Families}} for delta-model options.
For binomial family options, see 'Binomial families' in the Details section below.}

\item{space_columns}{A string or character vector that indicates
the column(s) of \code{data} indicating the location of each sample.
When \code{spatial_domain} is an \code{igraph} object, \code{space_columns} is a string with
with levels matching the names of vertices of that object.
When \code{spatial_domain} is an \code{fmesher} or \code{sfnetwork} object,
space_columns is a character vector indicating columns of \code{data} with
coordinates for each sample.}

\item{spatial_domain}{Object that represents spatial relationships, either using
\code{\link[fmesher:fm_mesh_2d]{fmesher::fm_mesh_2d()}} to apply the SPDE method,
\code{\link[igraph:make_empty_graph]{igraph::make_empty_graph()}} for independent time-series,
\code{\link[igraph:make_graph]{igraph::make_graph()}} to apply a simultaneous autoregressive (SAR)
process to a user-supplied graph, \code{\link[=sfnetwork_mesh]{sfnetwork_mesh()}} for stream networks,
or class \code{sfc_GEOMETRY} e.g constructed using \link[sf:st_make_grid]{sf::st_make_grid}
to apply a SAR to an areal model with adjacency
based on the geometry of the object,
or \code{NULL} to specify a single site.  If using \code{igraph} then the
graph must have vertex names \code{V(graph)$name} that match
levels of \code{data[,'space_columns']}}

\item{time_column}{A character string indicating the column of \code{data}
listing the time-interval for each sample, from the set of times
in argument \code{times}.}

\item{times}{A integer vector listing the set of times in order.
If \code{times=NULL}, then it is filled in as the vector of integers
from the minimum to maximum value of \code{data$time}.  Alternatively,
it could be the minimum value of \code{data$time} through future years,
such that the model can forecast those future years.}

\item{variable_column}{A character string indicating the column of \code{data}
listing the variable for each sample, from the set of times
in argument \code{variables}.}

\item{variables}{A character vector listing the set of variables.
if \code{variables=NULL}, then it is filled in as the unique values
from \code{data$variable_columns}.}

\item{distribution_column}{A character string indicating the column of \code{data}
listing the distribution for each sample, from the set of names
in argument \code{family}.
if \code{variables=NULL}, then it is filled in as the unique values
from \code{data$variables}.}

\item{delta_options}{a named list with slots for \code{formula},
\code{space_term}, and \code{spacetime_term}. These specify options for the
second linear predictor of a delta model, and are only used (or estimable)
when a \code{\link[tinyVAST:families]{delta family}} is used for some samples.}

\item{spatial_varying}{a formula specifying spatially varying coefficients.}

\item{weights}{A numeric vector representing optional likelihood weights for the
data likelihood. Weights do not have to sum to one and are not internally modified.
Thee weights argument needs to be a vector and not a name of the variable in the data frame.}

\item{control}{Output from \code{\link[=tinyVASTcontrol]{tinyVASTcontrol()}}, used to define user
settings.}
}
\value{
An object (list) of class \code{tinyVAST}. Elements include:
\describe{
\item{data}{Data-frame supplied during model fitting}
\item{spatial_domain}{the spatial domain supplied during fitting}
\item{formula}{the formula specified during model fitting}
\item{obj}{The TMB object from \code{\link[TMB]{MakeADFun}}}
\item{opt}{The output from \code{\link[stats]{nlminb}}}
\item{opt}{The report from \code{obj$report()}}
\item{sdrep}{The output from \code{\link[TMB]{sdreport}}}
\item{tmb_inputs}{The list of inputs passed to \code{\link[TMB]{MakeADFun}}}
\item{call}{A record of the function call}
\item{run_time}{Total time to run model}
\item{interal}{Objects useful for package function, i.e., all arguments
passed during the call}
\item{deviance_explained}{output from \code{\link{deviance_explained}}}
}
}
\description{
Fits a vector autoregressive spatio-temporal (VAST) model using
a minimal feature-set and a widely used interface.
}
\details{
\code{tinyVAST} includes several basic inputs that specify the model structure:
\itemize{
\item \code{formula} specifies covariates and splines in a Generalized Additive Model;
\item \code{time_term} specifies interactions among variables and over time that
are constant across space, constructing the time-variable interaction.
\item \code{space_term} specifies interactions among variables and over time that occur
based on the variable values at each location, constructing
the space-variable interaction.
\item \code{spacetime_term} specifies interactions among variables and over time, constructing
the space-time-variable interaction.
}

These inputs require defining the \emph{domain} of the model.  This includes:
\itemize{
\item \code{spatial_domain} specifies spatial domain, with determines spatial correlations
\item \code{times} specifies the temporal domain, i.e., sequence of time-steps
\item \code{variables} specifies the set of variables, i.e., the variables that will be modeled
}

The default \code{spacetime_term=NULL} and \code{space_term=NULL} turns off all multivariate
and temporal indexing, such that \code{spatial_domain} is then ignored, and the model collapses
to a generalized additive model using \code{\link[mgcv]{gam}}.  To specify a univariate spatial model,
the user must specify \code{spatial_domain} and either \code{space_term=""} or \code{spacetime_term=""}, where the latter
two are then parsed to include a single exogenous variance for the single variable\tabular{ll}{
   \strong{Model type} \tab \strong{How to specify} \cr
   Generalized additive model \tab specify \code{spatial_domain=NULL} \code{space_term=""} and \code{spacetime_term=""}, and then use \code{formula} to specify splines and covariates \cr
   Dynamic structural equation model (including vector autoregressive, dynamic factor analysis, ARIMA, and structural equation models) \tab specify \code{spatial_domain=NULL} and use \code{spacetime_term} to specify interactions among variables and over time \cr
   Univariate spatio-temporal model, or multiple independence spatio-temporal variables \tab specify \code{spatial_domain} and \code{spacetime_term=""}, where the latter is then parsed to include a single exogenous variance for the single variable \cr
   Multivariate spatial model including interactions \tab specify \code{spatial_domain} and use \code{space_term} to specify spatial interactions \cr
   Vector autoregressive spatio-temporal model (i.e., lag-1 interactions among variables) \tab specify \code{spatial_domain} and use \code{spacetime_term=""} to specify interactions among variables and over time, where spatio-temporal variables are constructed via the separable interaction of \code{spacetime_term} and \code{spatial_domain} \cr
}


\strong{Model building notes}
\itemize{
\item \verb{binomial familes}:  A binomial family can be specified in only one way:
the response is the observed proportion (proportion = successes / trials),
and the 'weights' argument is used to specify the Binomial size (trials, N)
parameter (\verb{proportion ~ ..., weights = N}).
\item \verb{factor models}:  If a factor model is desired, the factor(s) must be named
and included in the \code{variables}.  The factor is then modeled for \code{space_term},
\code{time_term}, and \code{spacetime_term} and it's variance must be fixed a priori
for any term where it is not being used.
}
}
\examples{
# Simulate a seperable two-dimensional AR1 spatial process
n_x = n_y = 25
n_w = 10
R_xx = exp(-0.4 * abs(outer(1:n_x, 1:n_x, FUN="-")) )
R_yy = exp(-0.4 * abs(outer(1:n_y, 1:n_y, FUN="-")) )
z = mvtnorm::rmvnorm(1, sigma=kronecker(R_xx,R_yy) )

# Simulate nuissance parameter z from oscillatory (day-night) process
w = sample(1:n_w, replace=TRUE, size=length(z))
Data = data.frame( expand.grid(x=1:n_x, y=1:n_y), w=w, z=as.vector(z) + cos(w/n_w*2*pi))
Data$n = Data$z + rnorm(nrow(Data), sd=1)

# Add columns for multivariate and/or temporal dimensions
Data$var = "n"

# make SPDE mesh for spatial term
mesh = fmesher::fm_mesh_2d( Data[,c('x','y')], n=100 )

# fit model with cyclic confounder as GAM term
out = tinyVAST( data = Data,
                formula = n ~ s(w),
                spatial_domain = mesh,
                space_term = "n <-> n, sd_n" )

# Run crossvalidation (too slow for CRAN)
\donttest{
CV = cv::cv( out, k = 4 )
CV
}

}
\seealso{
Details section of \code{\link[=make_dsem_ram]{make_dsem_ram()}} for a summary of the math involved with constructing the DSEM, and \doi{10.1111/2041-210X.14289} for more background on math and inference

\doi{10.48550/arXiv.2401.10193} for more details on how GAM, SEM, and DSEM components are combined from a statistical and software-user perspective

\code{\link[=summary.tinyVAST]{summary.tinyVAST()}} to visualize parameter estimates related to SEM and DSEM model components
}
