#' Spanning labels to identify groups of rows or columns
#'
#' @export
#' @inheritParams tt
#' @inheritParams style_tt
#' @param i A named list of row indices to group. The names of the list will be used as labels.
#' @param j A named list of column indices to group. The names of the list will be used as labels. See examples below. Note: empty labels must be a space: " ".
#' @param ... Other arguments are ignored.
#' @return An object of class `tt` representing the table.
#' @param indent integer number of `pt` to use when indenting the non-labelled rows.
#' @details 
#' Warning: The `style_tt()` can normally be used to style the group headers, as expected, but that feature is not available for Markdown and Word tables.
#' @examples 
#' 
#' tt(mtcars[1:10, 1:5]) |>
#'   group_tt(
#'     i = list(
#'       "Hello" = 3,
#'       "World" = 8),
#'     j = list(
#'       "Foo" = 2:3,
#'       "Bar" = 4:5))
group_tt <- function(x, i = NULL, j = NULL, indent = 1, ...) {
  # ... is important for ihead passing

  if (is.null(meta(x))) stop("`x` must be generated by `tinytable::tt()`.", call. = FALSE)
  if (is.null(i) && is.null(j)) stop("At least one of `i` or `j` must be specified.", call. = FALSE)
  assert_integerish(indent, lower = 0)

  out <- x

  i <- sanitize_group_index(i, hi = attr(x, "nrow") + 1, orientation = "row")
  j <- sanitize_group_index(j, hi = attr(x, "ncol"), orientation = "column")

  if (!is.null(i)) out <- meta(out, "nrows", meta(out, "nrows") + length(i))
  if (!is.null(j)) out <- meta(out, "nhead", meta(out, "nhead") + 1)
  
  # we don't need this as a list, and we use some sorting later
  i <- unlist(i)

  # the actual function is subbed in build_tt for html and grid
  cal <- call("group_tabularray", i = i, j = j, indent = indent)

  out <- meta(out, "lazy_group", c(meta(out)$lazy_group, list(cal)))

  return(out)
}



sanitize_group_index <- function(idx, hi, orientation) {
  if (is.null(idx)) return(idx)
  assert_list(idx, named = TRUE)
  for (n in names(idx)) {
    if (orientation == "row") {
      assert_integerish(idx[[n]], len = 1, lower = 1, upper = hi, name = n)
    } else {
      assert_integerish(idx[[n]], lower = 1, upper = hi, name = n)
    }
  }
  if (anyDuplicated(unlist(idx)) > 0) stop("Duplicate group indices.", call. = FALSE)
  out <- lapply(idx, function(x) min(x):max(x))
  return(out)
}
