#' Returns info about clicked rectangle
#'
#' This function returns information about the object on which user has clicked.
#' @param npcClick mouse click in npc coordinates, which can be generated by \code{\link{tmClick}}
#' @param tmSave treemap information: value returned by \code{\link{tmPlot}}
#' @export
#' @examples data(sbsData)
#' \dontrun{
#' tmSave <- tmPlot(sbsData, 
#'				 index="section", 
#'				 vSize="employees09+value added09+turnover09+salaries09",
#'				 vColor="employees08+value added08+turnover08+salaries08",
#'				 sortID="-size")
#'	# capture mouseclick
#'	npcClick <- tmClick()
#'	
#'	# locate clicked object
#'	print(tmLocate(npcClick, tmSave))
#'}
tmLocate <-
function(npcClick, tmSave) {
	
		# determine row, column and the treemap
		
		click_row <- tmSave$nRow - npcClick$y %/% (1/tmSave$nRow)
		click_col <- 1 + npcClick$x %/% (1/tmSave$nCol)

		index <- click_row + (click_col - 1) * tmSave$nRow
		
		if (index > length(tmSave$tm)) {
			stop("no treemap selected")
		}
		
		cat("row", click_row, ", col", click_col, ", treemap", index, "\n")

		# convert click to inches
		inchClick <- list()
		inchClick$x <- convertX(unit(npcClick$x, "npc"), "inches")
		inchClick$y <- convertY(unit(npcClick$y, "npc"), "inches")

		# zoom in to selected treemap
		downViewport("grid")
		downViewport(paste("tm",index,sep=""))
		downViewport("dataregion")
		downViewport("dataregion2")

		# determine treemap size in inches
		sizeTm <- c(convertWidth(unit(1, "npc"), "inches", valueOnly=TRUE),
              convertHeight(unit(1, "npc"), "inches", valueOnly=TRUE))

		# get transformation matrix
		trans <- current.transform()
		
		# go to root viewport
		upViewport(n=4)

		# calculate click (inches) relative to selected treemap
		tempCoor <- c(unlist(inchClick), 1) %*% solve(trans)
		inchCoor <- (tempCoor/tempCoor[3])[-3]

		# transform to npc coordinates
		npcCoor <- inchCoor / sizeTm
		
		cat("click: x y", npcCoor[1], npcCoor[2], "\n")
		
		# retrieve selected treemap
		tmsel <- tmSave$tm[[index]]
		
		# retrieve selected rectangle
		rectInd <- which(tmsel$x0 < npcCoor[1] & 
			(tmsel$x0 + tmsel$w) > npcCoor[1] & 
			tmsel$y0 < npcCoor[2] & 
			(tmsel$y0 + tmsel$h) > npcCoor[2])

		if (length(rectInd)==0) {
			stop("no treemap selected")
		}
		
		return(tmsel[rectInd[1], ])
}
