% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nlm.prob.R
\name{nlm.prob}
\alias{nlm.prob}
\title{Nonlinear minimisation/maximisation allowing probability vectors as inputs}
\usage{
nlm.prob(
  f,
  p,
  prob.vectors = list(1:length(p)),
  ...,
  lambda = 1,
  eta0max = 1e+10,
  maximise = FALSE,
  maximize = maximise,
  hessian = FALSE,
  typsize = rep(1, length(p)),
  fscale = 1,
  print.level = 0,
  ndigit = 12,
  gradtol = 1e-06,
  stepmax = max(1000 * sqrt(sum((p/typsize)^2)), 1000),
  steptol = 1e-06,
  iterlim = 100,
  check.analyticals = TRUE
)
}
\arguments{
\item{f}{The objective function to be minimised; output should be a single numeric value.}

\item{p}{Starting argument values for the minimisation.}

\item{prob.vectors}{A list specifying which sets of elements are constrained to be a probability vector (each element in the list
should be a vector specifying indices in the argument vector; elements cannot overlap into multiple probability vectors).}

\item{...}{Additional arguments to be passed to \code{f} via \code{nlm}}

\item{lambda}{The tuning parameter used in the softmax transformation for the optimisation (a single positive numeric value).}

\item{eta0max}{The maximum absolute value for the elements of eta0 (the starting value in the unconstrained optimisation problem).}

\item{maximise, maximize}{Logical value; if \code{TRUE} the function maximises the objective function instead of mimimising.}

\item{hessian}{Logical; if \code{TRUE} then the output of the function includes the Hessian of \code{f} at the minimising point.}

\item{typsize}{An estimate of the size of each parameter at the minimum.}

\item{fscale}{An estimate of the size of \code{f} at the minimum.}

\item{print.level}{This argument determines the level of printing which is done during the minimisation process. The default value
of \code{0} means that no printing occurs, a value of \code{1} means that initial and final details are printed and a value of \code{2}
means that full tracing information is printed.}

\item{ndigit}{The number of significant digits in the function \code{f}.}

\item{gradtol}{A positive scalar giving the tolerance at which the scaled gradient is considered close enough to zero to terminate
the algorithm. The scaled gradient is a measure of the relative change in \code{f} in each direction \code{p[i]} divided by the
relative change in \code{p[i]}.}

\item{stepmax}{A positive scalar which gives the maximum allowable scaled step length.  \code{stepmax} is used to prevent steps which
would cause the optimisation function to overflow, to prevent the algorithm from leaving the area of interest in parameter space, or
to detect divergence in the algorithm.  \code{stepmax} would be chosen small enough to prevent the first two of these occurrences,
but should be larger than any anticipated reasonable step.}

\item{steptol}{A positive scalar providing the minimum allowable relative step length.}

\item{iterlim}{A positive integer specifying the maximum number of iterations to be performed before the routine is terminated.}

\item{check.analyticals}{Logical; if \code{TRUE} then the analytic gradients and Hessians (if supplied) are checked against numerical
derivatives at the initial parameter values.  This can help detect incorrectly formulated gradients or Hessians.}
}
\value{
A list showing the computed minimising point and minimum of \code{f} and other related information.
}
\description{
\code{nlm.prob} minimises/maximises a function allowing probability vectors as inputs
}
\details{
This is a variation of the \code{stats::nlm} function for nonlinear minimisation.  The present function is designed to
minimise an objective function with one or more arguments that are probability vectors.  (The objective function may also
have other arguments that are not probability vectors.)  The function uses the same inputs as the \code{stats::nlm} function,
except that the user can use the input \code{prob.vectors} to specify which inputs are constrained to be probability vectors.
This input is a list where each element in the list specifies a set of indices for the argument of the objective function; the
specified set of indices is constrained to be a probability vector (i.e., each corresponding argument is non-negative and the
set of these arguments must sum to one).  The input \code{prob.vectors} may list one or more probability vectors, but they must
use disjoint elements of the argument (i.e., a variable in the argument cannot appear in more than one probability vector).

Optimisation is performed by first converting the objective function into unconstrained form using the softmax transformation
and its inverse to convert from unconstrained space to probability space and back.  Optimisation is done on the unconstrained
objective function and the results are converted back to probability space to solve the constrained optimisation problem.  For
purposes of conversion, this function allows specification of a tuning parameter \code{lambda} for the softmax and inverse-
softmax transformations.  (This input can either be a single tuning value used for all conversions, or a vector of values for
the respective probability vectors; if the latter, there must be one value for each element of the \code{prob.vector} input.)

Most of the input descriptions below are adapted from the corresponding descriptions in \code{stat::nlm}, since our function is
a wrapper to that function.  The additional inputs for this function are \code{prob.vectors}, \code{lambda} and \code{eta0max}.
The function also adds an option \code{maximise} to conduct maximisation instead of minimisation.
}
\examples{
x <- rbinom(100, 1, .2)
nlm.prob(function(p) sum(dbinom(x,1,p[2],log=TRUE)), c(.5, .5), maximise = TRUE)
}
