% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SVC_mle.R
\name{SVC_mle}
\alias{SVC_mle}
\alias{SVC_mle.default}
\alias{SVC_mle.formula}
\title{MLE of SVC model}
\usage{
SVC_mle(...)

\method{SVC_mle}{default}(y, X, locs, W = NULL, control = NULL,
  optim.control = list(), ...)

\method{SVC_mle}{formula}(formula, data, RE_formula = NULL, locs,
  control, optim.control = list(), ...)
}
\arguments{
\item{...}{further arguments}

\item{y}{numeric response vector of dimension n.}

\item{X}{matrix of covariates of dimension n x pX. Intercept has to be added manually.}

\item{locs}{matrix of locations of dimension n X 2. May contain multiple observations at single location which (may) cause a permutation of \code{y}, \code{X}, \code{W} and \code{locs}.}

\item{W}{Optional matrix of covariates with fixed effects, i.e. non-SVC, of dimension n x pW}

\item{control}{list of control paramaters, usually given by \code{\link{SVC_mle_control}}}

\item{optim.control}{list of control arguments for optimization function, see Details in \code{\link{optim}}}

\item{formula}{Formula describing the fixed effects in SVC model. The response, i.e. LHS of the formula, is not allowed to have functions such as \code{sqrt()} or \code{log()}.}

\item{data}{data frame containing the observations}

\item{RE_formula}{Formula describing the random effects in SVC model. Only RHS is considered. If \code{NULL}, the same RHS of argument \code{formula} for fixed effects is used.}
}
\value{
Object of class \code{SVC_mle}
}
\description{
Calls MLE of the SVC model defined as:

\deqn{y(s) = X \mu + W \eta (s) + \epsilon(s)}

where:
\itemize{
  \item y is the response (vector of length n)
  \item X is the data matrix for the fixed effects covariates
  \item \eqn{\mu} is the vetor containing the fixed effects
  \item W is the data matrix for the SVCs represented by zero mean GRF
  \item \eqn{\eta} are the SVCs represented by zero mean GRF
  \item \eqn{\epsilon} is the nugget effect
}

The MLE is done by calling the function \code{optim}.
}
\examples{
## ---- toy example ----
## sample data
# setting seed for reproducibility
set.seed(123)
m <- 7
# number of observations
n <- m*m
# number of SVC
p <- 3
# sample data
y <- rnorm(n)
X <- matrix(rnorm(n*p), ncol = p)
# locations on a regular m-by-m-grid
locs <- expand.grid(seq(0, 1, length.out = m),
                    seq(0, 1, length.out = m))

## preparing for maximum likelihood estimation (MLE)
# controls specific to MLE
control <- SVC_mle_control(
  # initial values of optimization
  init = rep(0.1, 2*p+1),
  # using profile likelihood
  profileLik = TRUE
)

# controls specific to optimization procedure, see help(optim)
opt.control <- list(
  # number of iterations (set to one for demonstration sake)
  maxit = 1,
  # tracing information
  trace = 6
)

## starting MLE
fit <- SVC_mle(y = y, X = X, locs = locs,
               control = control,
               optim.control = opt.control)

## output: convergence code equal to 1, since maxit was only 1
summary(fit)

\donttest{
## ---- real data example ----
require(sp)
## get data set
data("meuse", package = "sp")

# construct data matrix and response, scale locations
y <- log(meuse$cadmium)
X <- model.matrix(~1+dist+lime+elev, data = meuse)
locs <- as.matrix(meuse[, 1:2])/1000


## starting MLE
# the next call takes a couple of seconds
fit <- SVC_mle(y = y, X = X, locs = locs,
               # has 4 fixed effects, but only 3 random effects (SVC)
               # elev is missing in SVC
               W = X[, 1:3],
               control = SVC_mle_control(
                 # inital values for 3 SVC
                 # 7 = (3 * 2 covariance parameters + nugget)
                 init = c(rep(c(0.4, 0.2), 3), 0.2),
                 profileLik = TRUE
               ))

## summary and residual output
summary(fit)
plot(fit)

## predict
# new locations
newlocs <- expand.grid(
  x = seq(min(locs[, 1]), max(locs[, 1]), length.out = 30),
  y = seq(min(locs[, 2]), max(locs[, 2]), length.out = 30))
# predict SVC for new locations
SVC <- predict(fit, newlocs = as.matrix(newlocs))
# visualization
sp.SVC <- SVC
coordinates(sp.SVC) <- ~loc_x+loc_y
spplot(sp.SVC, colorkey = TRUE)
}
}
\seealso{
\code{\link{predict.SVC_mle}}
}
\author{
Jakob Dambon
}
