\encoding{UTF-8}
\name{oecosimu}
\alias{oecosimu}
\alias{summary.oecosimu}
\alias{as.ts.oecosimu}
\alias{toCoda}
\alias{toCoda.oecosimu}

\title{Evaluate Statistics with Null Models of Biological Communities }

\description{
 
  Function evaluates a statistic or a vector of statistics in
  community and evaluates its significance in a series of simulated
  random communities.  The approach has been used traditionally for
  the analysis of nestedness, but the function is more general and can
  be used with any statistics evaluated with simulated
  communities. Function \code{oecosimu} collects and evaluates the
  statistics. The Null model communities are described in
  \code{\link{make.commsim}} and \code{\link{permatfull}}/
  \code{\link{permatswap}}, the definition of Null models in
  \code{\link{nullmodel}}, and nestedness statistics in
  \code{\link{nestednodf}} (which describes several alternative
  statistics, including nestedness temperature, \eqn{N0}, checker
  board units, nestedness discrepancy and NODF).

}

\usage{
oecosimu(comm, nestfun, method, nsimul = 99, burnin = 0, thin = 1,
   statistic = "statistic", alternative = c("two.sided", "less", "greater"), 
   batchsize = NA, parallel = getOption("mc.cores"), ...)
\method{summary}{oecosimu}(object, ...)
\method{as.ts}{oecosimu}(x, ...)
\method{toCoda}{oecosimu}(x)
}

\arguments{
  \item{comm}{Community data, or a Null model object generated by
    \code{\link{nullmodel}} or an object of class \code{simmat} (array
    of permuted matrices from \code{\link{simulate.nullmodel}}). If
    \code{comm} is a community data, null model simulation
    \code{method} must be specified.  If \code{comm} is a
    \code{\link{nullmodel}}, the simulation \code{method} is ignored,
    and if \code{comm} is a \code{simmat} object, all other arguments
    are ignored except \code{nestfun}, \code{statistic} and
    \code{alternative}.}
  \item{nestfun}{Function analysed. Some nestedness functions are
    provided in \pkg{vegan} (see \code{\link{nestedtemp}}), but any
    function can be used if it accepts the community as the first
    argument, and returns either a plain number or a vector or the
    result in list item with the name defined in argument
    \code{statistic}. See Examples for defining your own functions.}
  \item{method}{Null model method: either a name (character string) of
    a method defined in \code{\link{make.commsim}} or a
    \code{\link{commsim}} function. This argument is ignored if
    \code{comm} is a \code{\link{nullmodel}} or a \code{simmat}
    object. See Details and Examples.}
  \item{nsimul}{Number of simulated null communities (ignored if
    \code{comm} is a \code{simmat} object).}
  \item{burnin}{Number of null communities discarded before proper
    analysis in sequential methods (such as \code{"tswap"})
    (ignored with non-sequential methods or when \code{comm} is a
    \code{simmat} object).}
  \item{thin}{Number of discarded null communities between two
    evaluations of nestedness statistic in sequential methods (ignored
    with non-sequential methods or when \code{comm} is a \code{simmat}
    object).}
  \item{statistic}{The name of the statistic returned by
    \code{nestfun}.} 
  \item{alternative}{a character string specifying the alternative
    hypothesis, must be one of \code{"two.sided"} (default), \code{"greater"}
    or \code{"less"}. Please note that the \eqn{p}-value of two-sided
    test is approximately two times higher than in the corresponding
    one-sided test (\code{"greater"} or \code{"less"} depending on the
    sign of the difference).}
  \item{batchsize}{Size in Megabytes of largest simulation object. If
    a larger structure would be produced, the analysis is broken
    internally into batches. With default \code{NA} the analysis is
    not broken into batches.  See Details.}
  \item{parallel}{Number of parallel processes or a predefined socket
    cluster.  With \code{parallel = 1} uses ordinary, non-parallel
    processing. The parallel processing is done with \pkg{parallel}
    package.  If you define a \code{nestfun} in Windows that needs other
    \R packages than \pkg{vegan} or \pkg{permute}, you must set up a
    socket cluster before the call. }
  \item{x, object}{An \code{oecosimu} result object.}

  \item{\dots}{Other arguments to functions.}
}

\details{
  
  Function \code{oecosimu} is a wrapper that evaluates a statistic
  using function given by \code{nestfun}, and then simulates a series
  of null models based on \code{nullmodel}, and evaluates the
  statistic on these null models. The \pkg{vegan} packages contains
  some nestedness functions that are described separately
  (\code{\link{nestedchecker}}, \code{\link{nesteddisc}},
  \code{\link{nestedn0}}, \code{\link{nestedtemp}},
  \code{\link{nestednodf}}), but many other functions can be used as
  long as they are meaningful with simulated communities.  An
  applicable function must return either the statistic as a plain
  number or a vector, or as a list element \code{"statistic"} (like
  \code{\link{chisq.test}}), or in an item whose name is given in the
  argument \code{statistic}.  The statistic can be a single number
  (like typical for a nestedness index), or it can be a vector. The
  vector indices can be used to analyse site (row) or species (column)
  properties, see \code{\link{treedive}} for an example. Raup-Crick
  index (\code{\link{raupcrick}}) gives an example of using a
  dissimilarities.

  The Null model type can be given as a name (quoted character string)
  that is used to define a Null model in \code{\link{make.commsim}}.
  These include all binary models described by Wright et al. (1998),
  Jonsson (2001), Gotelli & Entsminger (2003), \enc{Miklós}{Miklos} &
  Podani (2004), and some others. There are several quantitative Null
  models, such those discussed by Hardy (2008), and several that are
  unpublished (see \code{\link{make.commsim}},
  \code{\link{permatfull}}, \code{\link{permatswap}} for
  discussion). The user can also define her own \code{\link{commsim}}
  function (see Examples).

  Function works by first defining a \code{\link{nullmodel}} with
  given \code{\link{commsim}}, and then generating a series of
  simulated communities with \code{\link{simulate.nullmodel}}. A
  shortcut can be used for any of these stages and the input can be
  \enumerate{
    \item Community data (\code{comm}), Null model function
      (\code{nestfun}) and the number of simulations (\code{nsimul}).
    \item A \code{\link{nullmodel}} object and the number of
      simulations, and argument \code{method} is ignored.
    \item A three-dimensional array of simulated communities generated
      with \code{\link{simulate.nullmodel}}, and arguments
      \code{method} and \code{nsimul} are ignored.  
  }
  The last case allows analysing several statistics with the same
  simulations.

  The function first generates simulations with given
  \code{\link{nullmodel}} and then analyses these using the
  \code{nestfun}.  With large data sets and/or large number of
  simulations, the generated objects can be very large, and if the
  memory is exhausted, the analysis can become very slow and the
  system can become unresponsive. The simulation will be broken into
  several smaller batches if the simulated \code{\link{nullmodel}}
  objective will be above the set \code{batchsize} to avoid memory
  problems (see \code{\link{object.size}} for estimating the size of
  the current data set). The parallel processing still increases the
  memory needs.  The parallel processing is only used for evaluating
  \code{nestfun}.  The main load may be in simulation of the
  \code{\link{nullmodel}}, and \code{parallel} argument does not help
  there. Function \code{summary} is based on
  \code{\link{summary.permustats}} and returns information on
  permutations.

  Function \code{as.ts} transforms the simulated results of sequential
  methods into a time series or a \code{\link{ts}} object. This allows
  using analytic tools for time series in studying the sequences (see
  examples). Function \code{toCoda} transforms the simulated results
  of sequential methods into an \code{"mcmc"} object of the
  \CRANpkg{coda} package. The \pkg{coda} package provides functions for
  the analysis of stationarity, adequacy of sample size,
  autocorrelation, need of burn-in and much more for sequential
  methods, and summary of the results. Please consult the
  documentation of the \pkg{coda} package.

  Function \code{\link{permustats}} provides support to the standard
  \code{\link{density}}, \code{\link[lattice]{densityplot}},
  \code{\link{qqnorm}} and \code{\link[lattice]{qqmath}} functions for
  the simulated values.
}

\value{ 

  Function \code{oecosimu} returns an object of class
  \code{"oecosimu"}.  The result object has items \code{statistic} and
  \code{oecosimu}.  The \code{statistic} contains the complete object
  returned by \code{nestfun} for the original data.  The
  \code{oecosimu} component contains the following items:
  \item{statistic}{Observed values of the statistic.}
  \item{simulated}{Simulated values of the statistic.}
  \item{means}{Mean values of the statistic from simulations.}
  \item{z}{Standardized effect sizes (SES, a.k.a. the \eqn{z}-values)
     of the observed statistic based on simulations.}
  \item{pval}{The \eqn{P}-values of the statistic based on simulations.}
  \item{alternative}{The type of testing as given in argument \code{alternative}.}
  \item{method}{The \code{method} used in \code{\link{nullmodel}}.}
  \item{isSeq}{\code{TRUE} if \code{method} was sequential.}

}

\references{
  Hardy, O. J. (2008) 
  Testing the spatial phylogenetic structure of local communities: 
  statistical performances of different null models 
  and test statistics on a locally neutral community. 
  \emph{Journal of Ecology} 96, 914--926.

  Gotelli, N.J. & Entsminger, N.J. (2003). Swap algorithms in null model
  analysis. \emph{Ecology} 84, 532--535.

  Jonsson, B.G. (2001) A null model for randomization tests of
  nestedness in species assemblages. \emph{Oecologia} 127, 309--313.

  \enc{Miklós}{Miklos}, I. & Podani, J. (2004). Randomization of presence-absence
  matrices: comments and new algorithms. \emph{Ecology} 85, 86--92.

  Wright, D.H., Patterson, B.D., Mikkelson, G.M., Cutler, A. & Atmar,
  W. (1998). A comparative analysis of nested subset patterns of species
  composition. \emph{Oecologia} 113, 1--20.
}

\author{Jari Oksanen and Peter Solymos}

\note{
  If you wonder about the name of \code{oecosimu}, look at journal
  names in the References (and more in \code{\link{nestedtemp}}).  

  The internal structure of the function was radically changed in
  \pkg{vegan 2.2-0} with introduction of \code{\link{commsim}} and
  \code{\link{nullmodel}} and deprecation of
  \code{\link{commsimulator}}.
}

\seealso{Function \code{oecosimu} currently defines null models with
  \code{\link{commsim}} and generates the simulated null model
  communities with \code{\link{nullmodel}} and
  \code{\link{simulate.nullmodel}}. For other applications of
  \code{oecosimu}, see \code{\link{treedive}} and
  \code{\link{raupcrick}}.

  See also \code{\link{nestedtemp}} (that also discusses other
  nestedness functions) and \code{\link{treedive}} for another
  application.
}

\examples{
## Use the first eigenvalue of correspondence analysis as an index
## of structure: a model for making your own functions.
data(sipoo)
## Traditional nestedness statistics (number of checkerboard units)
oecosimu(sipoo, nestedchecker, "r0")
## sequential model, one-sided test, a vector statistic
out <- oecosimu(sipoo, decorana, "swap", burnin=100, thin=10, 
   statistic="evals", alt = "greater")
out
## Inspect the swap sequence as a time series object
plot(as.ts(out))
lag.plot(as.ts(out))
acf(as.ts(out))
## Density plot in lattice graphics
permulattice(permustats(out), "densityplot", as.table = TRUE, layout = c(1,4))
## Use quantitative null models to compare
## mean Bray-Curtis dissimilarities
data(dune)
meandist <- function(x) mean(vegdist(x, "bray"))
mbc1 <- oecosimu(dune, meandist, "r2dtable")
mbc1

## Define your own null model as a 'commsim' function: shuffle cells
## in each row
foo <- function(x, n, nr, nc, ...) {
   out <- array(0, c(nr, nc, n))
   for (k in seq_len(n))
      out[,,k] <- apply(x, 2, function(z) sample(z, length(z)))
   out
}
cf <- commsim("myshuffle", foo, isSeq = FALSE, binary = FALSE, 
   mode = "double")
oecosimu(dune, meandist, cf)

## Use pre-built null model
nm <- simulate(nullmodel(sipoo, "curveball"), 99)
oecosimu(nm, nestedchecker)
## Several chains of a sequential model -- this can be generalized
## for parallel processing (see ?smbind)
nm <- replicate(5, simulate(nullmodel(sipoo, "swap"), 99,
   thin=10, burnin=100), simplify = FALSE)
## nm is now a list of nullmodels: use smbind to combine these into one
## nullmodel with several chains
## IGNORE_RDIFF_BEGIN
nm <- smbind(nm, MARGIN = 3)
nm
oecosimu(nm, nestedchecker)
## IGNORE_RDIFF_END
## After this you can use toCoda() and tools in the coda package to
## analyse the chains (these will show that thin, burnin and nsimul are
## all too low for real analysis).
}
\keyword{ multivariate }
\keyword{ datagen }
\keyword{ nonparametric }


