% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/synmat.R
\name{synmat}
\alias{synmat}
\title{Synchrony matrices}
\usage{
synmat(
  dat,
  times,
  method,
  tsrange = c(0, Inf),
  nsurrogs = 1000,
  scale.min = 2,
  scale.max.input = NULL,
  sigma = 1.05,
  f0 = 1,
  weighted = TRUE,
  sigthresh = 0.95
)
}
\arguments{
\item{dat}{A locations (rows) x time (columns) matrix of measurements}

\item{times}{The times at which measurements were made, spacing 1}

\item{method}{Method for synchrony calculation. See details.}

\item{tsrange}{A vector containing the min and max of the focal timescale range. Defaults 
to all timescales that are valid given choices for scale.min, scale.max.input, f0, sigma.
Only used for wavelet-based methods.}

\item{nsurrogs}{Number of surrogates for significance test. Defaults to 1000. Only used
for surrogate-based methods.}

\item{scale.min}{The smallest scale of fluctuation that will be examined. At least 2. Used 
only for wavelet-based methods.}

\item{scale.max.input}{The largest scale of fluctuation guaranteed to be examined. Only used 
for wavelet-based methods.}

\item{sigma}{The ratio of each time scale examined relative to the next timescale. Should be 
greater than 1. Only used for wavelet-based methods.}

\item{f0}{The ratio of the period of fluctuation to the width of the envelope. Only used for 
wavelet-based methods.}

\item{weighted}{If \code{TRUE}, create a weighted network. If \code{FALSE}, create a binary 
network using statistical significance. Binary networks are only allowed for networks based
on significance.}

\item{sigthresh}{Significance threshold needed, if \code{weighted} is false, for a network
link to be realized. Typically 0.95, 0.99, or 0.999, etc. Only used if \code{weighted} is
\code{FALSE}.}
}
\value{
\code{synmat} returns a synchrony matrix, of type depending on the \code{method}
argument. See details. Diagonal entries are left as \code{NA}.
}
\description{
Calculate synchrony matrices using a variety of methods
}
\details{
The following values are valid for \code{method}: 
\code{"pearson"}, \code{"pearson.sig.std"}, \code{"pearson.sig.fft"}, 
\code{"pearson.sig.aaft"}, 
\code{"spearman"}, \code{"spearman.sig.std"}, \code{"spearman.sig.fft"}, 
\code{"spearman.sig.aaft"}, 
\code{"kendall"}, \code{"kendall.sig.std"}, \code{"kendall.sig.fft"}, 
\code{"kendall.sig.aaft"}, 
\code{"ReXWT"}, \code{"ReXWT.sig.fft"}, \code{"ReXWT.sig.aaft"}, \code{"ReXWT.sig.fast"}, 
\code{"coh"}, \code{"coh.sig.fft"}, \code{"coh.sig.aaft"}, \code{"coh.sig.fast"},
\code{"phasecoh"}, \code{"phasecoh.sig.fft"}, and \code{"phasecoh.sig.aaft"}.
The first portions of these identifiers correspond to the Pearson, Spearman, and Kendall 
correlations, the real part of the cross-wavelet transform, the wavelet coherence, and the 
wavelet phase coherence. The second portions of these identifiers, when present, indicates
that significance of the measure specified in the first portion of the identifies is to
be used for establishing the synchrony matrix. Otherwise the value itself is used. The
third part of the \code{method} identifier indicates what type of significance is used.

Significance testing is performed using standard approaches (\code{method} flag containg
\code{std}; for correlation coefficients, 
although these are inappropriate for autocorrelated data), or surrogates generated using the 
Fourier (\code{method} flag containing \code{"fft"}) or amplitude adjusted Fourier 
surrogates (\code{"aaft"}). For 
\code{"coh"} and \code{"ReXWT"}, the fast testing algorithm of Sheppard et al. (2017) is also
implemented (\code{"fast"}). That method uses implicit Fourier surrogates. The choice of 
wavelet coherence (method flag containing \code{"coh"}) or the real part of 
the cross-wavelet 
transform (method flag containing \code{"ReXWT"}) depends mainly 
on treatment of out-of-phase 
relationships. The \code{"ReXWT"} is more akin to a correlation coefficient in that 
strong in-phase relationships approach 1 and strong antiphase relationships approach -1. 
Wavelet coherence allows any phase relationship and ranges from 0 to 1. Power normalization
is applied for \code{"coh"} and for \code{"ReXWT"}. All significance tests are one-tailed. 
Synchrony matrices for significance-based methods when \code{weighted} is \code{TRUE} 
contain 1 minus the p-values.
}
\examples{
sig<-matrix(.9,5,5)
diag(sig)<-1
if (requireNamespace("mvtnorm",quietly=TRUE))
{
  dat1<-t(mvtnorm::rmvnorm(30,mean=rep(0,5),sigma=sig))
  dat2<-t(mvtnorm::rmvnorm(30,mean=rep(0,5),sigma=sig))
}else
{
  dat1<-t(matrix(rep(rnorm(30),times=5),30,5))
  dat2<-t(matrix(rep(rnorm(30),times=5),30,5))
}
dat<-rbind(dat1,dat2)
times<-1:30
dat<-cleandat(dat,times,clev=2)$cdat
method<-"pearson.sig.fft"
res<-synmat(dat,times,method,nsurrogs=100,weighted=FALSE,
            sigthresh=0.95)
  
}
\references{
Walter, J. A., et al. (2017) The geography of spatial synchrony. Ecology 
Letters. doi: 10.1111/ele.12782
}
\seealso{
\code{\link{clust}}, \code{\link{coh}}, \code{\link{surrog}}, \code{browseVignettes("wsyn")}
}
\author{
Jonathan Walter, \email{jaw3es@virginia.edu}; Daniel Reuman, \email{reuman@ku.edu};
Lei Zhao, \email{lei.zhao@cau.edu.cn}
}
