#' Derive the PFI rank of all predictors used in the GRNN 
#'
#' The function \code{grnn.pfi} derives the PFI rank of all predictors used in the GRNN
#' It essentially is a wrapper around the function \code{grnn.x_pfi}.
#'
#' @param net   The GRNN object generated by grnn.fit() 
#' @param class TRUE or FALSE, whether it is for the classification or not
#' @param ntry  The number of random permutations to try, 1e3 times by default
#' @param seed  The seed value for the random permutation
#'
#' @return A dataframe with PFI values of all predictors in the GRNN
#'
#' @seealso \code{\link{grnn.x_pfi}}
#'
#' @examples
#' data(iris, package = "datasets")
#' Y <- ifelse(iris[, 5] == "setosa", 1, 0)
#' X <- scale(iris[, 1:3])
#' gnet <- grnn.fit(x = X, y = Y)
#' \dontrun{
#' grnn.pfi(net = gnet, class = TRUE)
#' }

grnn.pfi <- function(net, class = FALSE, ntry = 1e3, seed = 1) {
  if (class(net) != "General Regression Neural Net") stop("net needs to be a GRNN.", call. = F)
  if (!(class %in% c(TRUE, FALSE))) stop("the class input is not correct.", call. = F)

  cls <- parallel::makeCluster(min(ncol(net$x), parallel::detectCores() - 1), type = "PSOCK")
  obj <- c("net", "class", "grnn.fit", "grnn.predone", "grnn.predict", "grnn.x_pfi", "ntry", "seed")
  parallel::clusterExport(cls, obj,  envir = environment())
  rst1 <- data.frame(idx = seq(ncol(net$x)),
                     Reduce(rbind, parallel::parLapply(cls, seq(ncol(net$x)),
                                                       function(i) grnn.x_pfi(net, i, class = class, ntry = ntry, seed = seed))))
  parallel::stopCluster(cls)
  rst2 <- rst1[with(rst1, order(-pfi)), ]
  row.names(rst2) <- NULL
  return(rst2)
}
