% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_APCI.R
\name{fit_APCI}
\alias{fit_APCI}
\title{A function to fit the APCI stochastic mortality model.}
\usage{
fit_APCI(
  death,
  expo,
  n_iter = 10000,
  family = "nb",
  share_alpha = FALSE,
  share_beta = FALSE,
  share_gamma = FALSE,
  n.chain = 1,
  thin = 1,
  n.adapt = 1000,
  forecast = FALSE,
  h = 5,
  quiet = FALSE
)
}
\arguments{
\item{death}{death data that has been formatted through the function \code{preparedata_fn}.}

\item{expo}{expo data that has been formatted through the function \code{preparedata_fn}.}

\item{n_iter}{number of iterations to run. Default is \code{n_iter=10000}.}

\item{family}{a string of characters that defines the family function associated with the mortality model. "poisson" would assume that deaths follow a Poisson distribution and use a log link; "binomial" would assume that deaths follow a Binomial distribution and a logit link; "nb" (default) would assume that deaths follow a Negative-Binomial distribution and a log link.}

\item{share_alpha}{a logical value indicating if \eqn{a_{x,p}} should be shared across all strata (see details below). Default is \code{FALSE}.}

\item{share_beta}{a logical value indicating if \eqn{b_{x,p}} should be shared across all strata (see details below). Default is \code{FALSE}.}

\item{share_gamma}{a logical value indicating if the cohort parameter \eqn{\gamma_{x,p}} should be shared across all strata (see details below). Default is \code{FALSE}.}

\item{n.chain}{number of parallel chains for the model.}

\item{thin}{thinning interval for monitoring purposes.}

\item{n.adapt}{the number of iterations for adaptation. See \code{?rjags::adapt} for details.}

\item{forecast}{a logical value indicating if forecast is to be performed (default is \code{FALSE}). See below for details.}

\item{h}{a numeric value giving the number of years to forecast. Default is \code{h=5}.}

\item{quiet}{if TRUE then messages generated during compilation will be suppressed, as well as the progress bar during adaptation.}
}
\value{
A list with components:
\describe{
\item{\code{post_sample}}{An \code{mcmc.list} object containing the posterior samples generated.}
\item{\code{param}}{A vector of character strings describing the names of model parameters.}
\item{\code{death}}{The death data that was used.}
\item{\code{expo}}{The expo data that was used.}
\item{\code{family}}{The family function used.}
\item{\code{forecast}}{A logical value indicating if forecast has been performed.}
\item{\code{h}}{The forecast horizon used.}
}
}
\description{
Carry out Bayesian estimation of the stochastic mortality model, the \bold{Age-Period-Cohort-Improvement (APCI) model}.
Note that this model has been studied extensively by Richards et al. (2019) and Wong et al. (2023).
}
\details{
The model can be described mathematically as follows:
If \code{family="poisson"}, then
\deqn{d_{x,t,p} \sim \text{Poisson}(E^c_{x,t,p} m_{x,t,p}) , }
\deqn{\log(m_{x,t,p})=a_{x,p}+b_{x,p}(t-\bar{t})+k_{t,p} + \gamma_{c,p} , }
where \eqn{c=t-x} is the cohort index, \eqn{\bar{t}} is the mean of \eqn{t},
\eqn{d_{x,t,p}} represents the number of deaths at age \eqn{x} in year \eqn{t} of stratum \eqn{p},
while \eqn{E^c_{x,t,p}} and \eqn{m_{x,t,p}} represents respectively the corresponding central exposed to risk and central mortality rate at age \eqn{x} in year \eqn{t} of stratum \eqn{p}.
Similarly, if \code{family="nb"}, then a negative binomial distribution is fitted, i.e.
\deqn{d_{x,t,p} \sim \text{Negative-Binomial}(\phi,\frac{\phi}{\phi+E^c_{x,t,p} m_{x,t,p}}) , }
\deqn{\log(m_{x,t,p})=a_{x,p}+b_{x,p}(t-\bar{t})+k_{t,p} + \gamma_{c,p} , }
where \eqn{\phi} is the overdispersion parameter. See Wong et al. (2018).
But if \code{family="binomial"}, then
\deqn{d_{x,t,p} \sim \text{Binomial}(E^0_{x,t,p} , q_{x,t,p}) , }
\deqn{\text{logit}(q_{x,t,p})=a_{x,p}+b_{x,p}(t-\bar{t})+k_{t,p} + \gamma_{c,p} , }
where \eqn{q_{x,t,p}} represents the initial mortality rate at age \eqn{x} in year \eqn{t} of stratum \eqn{p},
while \eqn{E^0_{x,t,p}\approx E^c_{x,t,p}+\frac{1}{2}d_{x,t,p}} is the corresponding initial exposed to risk.
Constraints used are:
\deqn{ \sum_{t} k_{t,p} = \sum_{t} tk_{t,p} = 0, \sum_{c} \gamma_{c,p} = \sum_{c}c\gamma_{c,p} = \sum_{c}c^2\gamma_{c,p} = 0 \text{\ \ \ for each stratum }p.}
If \code{share_alpha=TRUE}, then the additive age-specific parameter is the same across all strata \eqn{p}, i. e.
\deqn{a_{x}+b_{x,p}(t-\bar{t})+k_{t,p}+ \gamma_{c,p} .}
Similarly if \code{share_beta=TRUE}, then the multiplicative age-specific parameter is the same across all strata \eqn{p}, i. e.
\deqn{a_{x,p}+b_{x}(t-\bar{t})+k_{t,p}+ \gamma_{c,p} .}
Similarly if \code{share_gamma=TRUE}, then the cohort parameter is the same across all strata \eqn{p}, i. e.
\deqn{a_{x,p}+b_{x,p}(t-\bar{t})+k_{t,p}+ \gamma_{c} .}
If \code{forecast=TRUE}, then the following time series models are fitted on \eqn{k_{t,p}} and \eqn{\gamma_{c,p}} as follows:
\deqn{k_{t,p} = \rho k_{t-1,p} + \epsilon_{t,p} \text{ for }p=1,\ldots,P \text{ and } t=2,\ldots,T,}
\deqn{k_{1,p}=\epsilon_{1,p}}
and
\deqn{\gamma_{c,p} = \gamma_{c-1,p}+ \rho_\gamma (\gamma_{c-1,p}-\gamma_{c-2,p}) + \epsilon^\gamma_{c,p} \text{ for }p=1,\ldots,P \text{ and } c=3,\ldots,C,}
\deqn{\gamma_2=\gamma_1+\frac{1}{\sqrt{1-\rho_\gamma^2}}\epsilon^\gamma_{2,p},}
\deqn{\gamma_1=100\epsilon^\gamma_{1,p}}
where \eqn{\epsilon_{t,p}\sim N(0,\sigma_k^2)} for \eqn{t=1,\ldots,T}, \eqn{\epsilon^\gamma_{c,p}\sim N(0,\sigma_\gamma^2)} for \eqn{c=1,\ldots,C}, while \eqn{\eta_1,\eta_2,\rho,\sigma_k^2,\rho_\gamma,\sigma_\gamma^2} are additional parameters to be estimated.
Note that the forecasting models are inspired by Wong et al. (2023).
}
\examples{
#load and prepare mortality data
data("dxt_array_product");data("Ext_array_product")
death<-preparedata_fn(dxt_array_product,strat_name = c("ACI","DB","SCI"),ages=35:65)
expo<-preparedata_fn(Ext_array_product,strat_name = c("ACI","DB","SCI"),ages=35:65)

#fit the model (negative-binomial family)
#NOTE: This is a toy example, please run it longer in practice.
fit_APCI_result<-fit_APCI(death=death,expo=expo,n_iter=50,n.adapt=50)
head(fit_APCI_result)

\donttest{
#if sharing the cohorts only (poisson family)
fit_APCI_result2<-fit_APCI(death=death,expo=expo,n_iter=1000,family="poisson",share_gamma=TRUE)
head(fit_APCI_result2)

#if sharing all alphas, betas, and cohorts (poisson family)
fit_APCI_result3<-fit_APCI(death=death,expo=expo,n_iter=1000,family="poisson",share_alpha=TRUE,
share_beta=TRUE,share_gamma=TRUE)
head(fit_APCI_result3)

#if forecast
fit_APCI_result4<-fit_APCI(death=death,expo=expo,n_iter=1000,family="poisson",forecast=TRUE)
plot_rates_fn(fit_APCI_result4)
plot_param_fn(fit_APCI_result4)
}
}
\references{
Richards S. J., Currie I. D., Kleinow T., and Ritchie G. P. (2019). A stochastic implementation of the APCI model for mortality projections. British Actuarial Journal, 24, E13. \doi{https://doi.org/10.1017/S1357321718000260}

Jackie S. T. Wong, Jonathan J. Forster, and Peter W. F. Smith. (2018). Bayesian mortality forecasting with overdispersion, Insurance: Mathematics and Economics, Volume 2018, Issue 3, 206-221. \doi{https://doi.org/10.1016/j.insmatheco.2017.09.023}

Jackie S. T. Wong, Jonathan J. Forster, and Peter W. F. Smith. (2023). Bayesian model comparison for mortality forecasting, Journal of the Royal Statistical Society Series C: Applied Statistics, Volume 72, Issue 3, 566–586. \doi{https://doi.org/10.1093/jrsssc/qlad021}
}
\concept{APCI}
\concept{parameter estimation}
\concept{stochastic mortality models}
\keyword{bayesian}
\keyword{estimation}
\keyword{models}
