% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glasso.R
\name{CVglasso}
\alias{CVglasso}
\title{Penalized precision matrix estimation}
\usage{
CVglasso(X = NULL, S = NULL, nlam = 10, lam.min.ratio = 0.01,
  lam = NULL, diagonal = FALSE, path = FALSE, tol = 1e-04,
  maxit = 10000, adjmaxit = NULL, K = 5, crit.cv = c("loglik", "AIC",
  "BIC"), start = c("warm", "cold"), cores = 1, trace = c("progress",
  "print", "none"), ...)
}
\arguments{
\item{X}{option to provide a nxp data matrix. Each row corresponds to a single observation and each column contains n observations of a single feature/variable.}

\item{S}{option to provide a pxp sample covariance matrix (denominator n). If argument is \code{NULL} and \code{X} is provided instead then \code{S} will be computed automatically.}

\item{nlam}{number of \code{lam} tuning parameters for penalty term generated from \code{lam.min.ratio} and \code{lam.max} (automatically generated). Defaults to 10.}

\item{lam.min.ratio}{smallest \code{lam} value provided as a fraction of \code{lam.max}. The function will automatically generate \code{nlam} tuning parameters from \code{lam.min.ratio*lam.max} to \code{lam.max} in log10 scale. \code{lam.max} is calculated to be the smallest \code{lam} such that all off-diagonal entries in \code{Omega} are equal to zero (\code{alpha} = 1). Defaults to 1e-2.}

\item{lam}{option to provide positive tuning parameters for penalty term. This will cause \code{nlam} and \code{lam.min.ratio} to be disregarded. If a vector of parameters is provided, they should be in increasing order. Defaults to NULL.}

\item{diagonal}{option to penalize the diagonal elements of the estimated precision matrix (\eqn{\Omega}). Defaults to \code{FALSE}.}

\item{path}{option to return the regularization path. This option should be used with extreme care if the dimension is large. If set to TRUE, cores must be set to 1 and errors and optimal tuning parameters will based on the full sample. Defaults to FALSE.}

\item{tol}{convergence tolerance. Iterations will stop when the average absolute difference in parameter estimates in less than \code{tol} times multiple. Defaults to 1e-4.}

\item{maxit}{maximum number of iterations. Defaults to 1e4.}

\item{adjmaxit}{adjusted maximum number of iterations. During cross validation this option allows the user to adjust the maximum number of iterations after the first \code{lam} tuning parameter has converged. This option is intended to be paired with \code{warm} starts and allows for 'one-step' estimators. Defaults to NULL.}

\item{K}{specify the number of folds for cross validation.}

\item{crit.cv}{cross validation criterion (\code{loglik}, \code{AIC}, or \code{BIC}). Defaults to \code{loglik}.}

\item{start}{specify \code{warm} or \code{cold} start for cross validation. Default is \code{warm}.}

\item{cores}{option to run CV in parallel. Defaults to \code{cores = 1}.}

\item{trace}{option to display progress of CV. Choose one of \code{progress} to print a progress bar, \code{print} to print completed tuning parameters, or \code{none}.}

\item{...}{additional arguments to pass to \code{glasso}.}
}
\value{
returns class object \code{CVglasso} which includes:
\item{Call}{function call.}
\item{Iterations}{number of iterations}
\item{Tuning}{optimal tuning parameters (lam and alpha).}
\item{Lambdas}{grid of lambda values for CV.}
\item{maxit}{maximum number of iterations for outer (blockwise) loop.}
\item{Omega}{estimated penalized precision matrix.}
\item{Sigma}{estimated covariance matrix from the penalized precision matrix (inverse of Omega).}
\item{Path}{array containing the solution path. Solutions will be ordered by ascending lambda values.}
\item{MIN.error}{minimum average cross validation error (cv.crit) for optimal parameters.}
\item{AVG.error}{average cross validation error (cv.crit) across all folds.}
\item{CV.error}{cross validation errors (cv.crit).}
}
\description{
Penalized precision matrix estimation using the graphical lasso (glasso) algorithm.
Consider the case where \eqn{X_{1}, ..., X_{n}} are iid \eqn{N_{p}(\mu,
\Sigma)} and we are tasked with estimating the precision matrix,
denoted \eqn{\Omega \equiv \Sigma^{-1}}. This function solves the
following optimization problem:
\describe{
\item{Objective:}{
\eqn{\hat{\Omega}_{\lambda} = \arg\min_{\Omega \in S_{+}^{p}}
\left\{ Tr\left(S\Omega\right) - \log \det\left(\Omega \right) +
\lambda \left\| \Omega \right\|_{1} \right\}}}
}
where \eqn{\lambda > 0} and we define
\eqn{\left\|A \right\|_{1} = \sum_{i, j} \left| A_{ij} \right|}.
}
\details{
For details on the implementation of the 'glasso' function, see Tibshirani's website.
\url{http://statweb.stanford.edu/~tibs/glasso/}.
}
\examples{
# generate data from a sparse matrix
# first compute covariance matrix
S = matrix(0.7, nrow = 5, ncol = 5)
for (i in 1:5){
 for (j in 1:5){
   S[i, j] = S[i, j]^abs(i - j)
 }
}

# generate 100 x 5 matrix with rows drawn from iid N_p(0, S)
Z = matrix(rnorm(100*5), nrow = 100, ncol = 5)
out = eigen(S, symmetric = TRUE)
S.sqrt = out$vectors \%*\% diag(out$values^0.5)
S.sqrt = S.sqrt \%*\% t(out$vectors)
X = Z \%*\% S.sqrt

# lasso penalty CV
CVglasso(X)
}
\references{
\itemize{
\item Friedman, Jerome, Trevor Hastie, and Robert Tibshirani. 'Sparse inverse covariance estimation with the graphical lasso.' \emph{Biostatistics} 9.3 (2008): 432-441.
\item Banerjee, Onureen, Ghauoui, Laurent El, and d'Aspremont, Alexandre. 2008. 'Model Selection through Sparse Maximum Likelihood Estimation for Multivariate Gaussian or Binary Data.' \emph{Journal of Machine Learning Research} 9: 485-516.
\item Tibshirani, Robert. 1996. 'Regression Shrinkage and Selection via the Lasso.' \emph{Journal of the Royal Statistical Society. Series B (Methodological)}. JSTOR: 267-288.
\item Meinshausen, Nicolai and Buhlmann, Peter. 2006. 'High-Dimensional Graphs and Variable Selection with the Lasso.' \emph{The Annals of Statistics}. JSTOR: 1436-1462.
\item Witten, Daniela M, Friedman, Jerome H, and Simon, Noah. 2011. 'New Insights and Faster computations for the Graphical Lasso.' \emph{Journal of Computation and Graphical Statistics}. Taylor and Francis: 892-900.
\item Tibshirani, Robert, Bien, Jacob, Friedman, Jerome, Hastie, Trevor, Simon, Noah, Jonathan, Taylor, and Tibshirani, Ryan J. 'Strong Rules for Discarding Predictors in Lasso-Type Problems.' \emph{Journal of the Royal Statistical Society: Series B (Statistical Methodology)}. Wiley Online Library 74 (2): 245-266.
\item Ghaoui, Laurent El, Viallon, Vivian, and Rabbani, Tarek. 2010. 'Safe Feature Elimination for the Lasso and Sparse Supervised Learning Problems.' \emph{arXiv preprint arXiv: 1009.4219}.
\item Osborne, Michael R, Presnell, Brett, and Turlach, Berwin A. 'On the Lasso and its Dual.' \emph{Journal of Computational and Graphical Statistics}. Taylor and Francis 9 (2): 319-337.
\item Rothman, Adam. 2017. 'STAT 8931 notes on an algorithm to compute the Lasso-penalized Gausssian likelihood precision matrix estimator.' 
}
}
\seealso{
\code{\link{plot.CVglasso}}
}
\author{
Matt Galloway \email{gall0441@umn.edu}
}
