#' Dual Wavelet-based NARDL Model
#'
#' This function implements the Dual Wavelet-based Nonlinear Autoregressive Distributed Lag (NARDL) model.
#'
#' @param Data A time series object (numeric vector) representing the dependent variable.
#' @param Exo A time series object (numeric vector) representing the exogenous variable.
#' @param MaxLag Maximum number of lags to consider.
#' @param Trend Boolean to include trend in the model. Default is TRUE.
#'
#' @return A list containing:
#' \item{Coefficients}{Model coefficients (short and long run).}
#' \item{AsymTest}{Wald test statistics and p-values.}
#' \item{IC}{Information criteria (AIC, BIC, Log-likelihood).}
#'
#' @references
#' Jammazi, R., Lahiani, A., & Nguyen, D. K. (2015). A wavelet-based nonlinear ARDL model for assessing the exchange rate pass-through to crude oil prices. *Journal of International Financial Markets, Institutions and Money, 34*, 173-187. https://doi.org/10.1016/j.intfin.2014.11.011
#'
#' @examples
#' Data <- rnorm(100)
#' Exo <- rnorm(100)
#' Results <- NARDL(Data, Exo, MaxLag = 3)
NARDL <- function(Data, Exo, MaxLag, Trend = TRUE) {
  Data1 <- as.ts(Data)
  Data2 <- as.ts(Exo)

  if (Trend == TRUE) {
    t = 5
  } else if (Trend == FALSE) {
    t = 3
  }

  Data_ARDL <- cbind.data.frame(Data1, Data2)
  colnames(Data_ARDL) <- c("y", "x")

  NARDL <- nardl(y ~ x, Data_ARDL, ic = "bic", maxlag = MaxLag, graph = TRUE, case = t)
  Res <- rbind(NARDL$sels$coefficients, NARDL$lres)

  Wald <- rbind.data.frame(NARDL$wldsr, NARDL$wldq)
  colnames(Wald) <- c("Statistic", "pvalue")
  rownames(Wald) <- c("Short Run", "Long Run")

  IC <- cbind.data.frame(AIC = AIC(NARDL$fits), BIC = BIC(NARDL$fits), LogLik = logLik(NARDL$fits)[1])

  Results <- list(Coefficients = Res, AsymTest = Wald, IC = IC)
  return(Results)
}
