#' High-Dimensional Two-Sample Mean Test
#'
#' Conducts a high-dimensional two-sample mean test using a modified Hotelling's T-squared statistic.
#' This test is suitable for cases where the number of variables \( p \) is larger than the sample size \( n \).
#'
#' @param X Matrix representing the first sample (rows are observations, columns are variables).
#' @param Y Matrix representing the second sample (rows are observations, columns are variables).
#' @return A list containing:
#'   \item{TSvalue}{The test statistic value.}
#'   \item{pvalue}{The p-value of the test.}
#'
#' @details This function implements a high-dimensional two-sample mean test by adjusting the Hotelling's T-squared statistic.
#' It uses diagonal matrices and a correction factor to handle high-dimensional data.
#' @examples
#' # Example usage:
#' set.seed(123)
#' X <- matrix(rnorm(200), nrow = 10, ncol = 20)  # 10 samples, 20 variables
#' Y <- matrix(rnorm(200, mean = 0.5), nrow = 10, ncol = 20)  # Different mean
#' result <- SKK(X, Y)
#' print(result)
#' @export
SKK<-  function(X, Y)
{
   n1 <- nrow(X)
   n2 <- nrow(Y)
   p <- ncol(X)
   n <- n1 + n2 - 2
   X.bar <- apply(X, 2, mean)
   Y.bar <- apply(Y, 2, mean)
   S1 <- stats::cov(X)
   S2 <- stats::cov(Y)
   D1 <- diag(diag(S1))
   D2 <- diag(diag(S2))
   D <- D1 / n1 + D2 / n2 
   R <- solve(sqrt(D)) %*% (S1 / n1 + S2 / n2) %*%  solve(sqrt(D)) # 
   c.p.n <- 1 + sum(diag(R %*% R) / p ^ (3 / 2))#
   var.qn <-
      2 * sum(diag(R %*% R)) / p - 2 * sum(diag(solve(D) %*% S1)) ^ 2 / p / n1 /
      (n1 + 1) ^ 2 - 2 * sum(diag(solve(D) %*% S2)) ^ 2 / p / n2 / (n2 + 1) ^ 2#
   denom <- sqrt(p * c.p.n * var.qn)   #
   TSvalue <-
      (t(X.bar - Y.bar) %*% solve(D) %*% (X.bar - Y.bar) - p) / denom #
   pvalue <- 2 * (1 - stats::pnorm(abs(TSvalue), 0, 1))
   Tn = t(X.bar - Y.bar) %*% solve(D) %*% (X.bar - Y.bar)/p #
   list(TSvalue = TSvalue, pvalue = pvalue)
}
