#' Random Perturbation Robust PCA
#'
#' Robust PCA by adding random symmetric perturbation to covariance matrix.
#'
#' @param data T x N matrix.
#' @param m Components.
#' @param epsilon Perturbation scale.
#' @param sigma Noise sd.
#' @return List with Ap, Dp, lambda.
#' @export
#' @examples
#' \dontrun{
#' set.seed(456)
#' N <- 80; T <- 100; m.true <- 2
#' X <- matrix(rnorm(T * N), T, N)
#' rpca.out <- EFM::robust_perturb_pca(X, m = m.true)
#' print(round(rpca.out$Ap[1:5, 1:2], 3))
#' }
robust_perturb_pca <- function(data, m, epsilon = 1e-3, sigma = 0.01) {
  X <- scale(data); n <- nrow(X); p <- ncol(X)
  S <- cov(X)
  Delta <- matrix(rnorm(p * p, 0, sigma), p, p)
  Delta <- (Delta + t(Delta)) / 2
  Se <- S + epsilon * Delta
  eig <- eigen(Se)
  ind <- order(eig$values, decreasing = TRUE)[1:m]
  Vm <- eig$vectors[, ind, drop = FALSE]
  lam <- eig$values[ind]
  Ap <- Vm %*% diag(sqrt(lam))
  h2 <- diag(Ap %*% t(Ap))
  Dp <- diag(S) - h2
  list(Ap = Ap, Dp = Dp, lambda = lam)
}