% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/render_erd.R
\name{render_erd}
\alias{render_erd}
\title{Render ERD}
\usage{
render_erd(erd_object, label_distance = 2.5, label_angle = 45, n = 10)
}
\arguments{
\item{erd_object}{An object of class "ERD", generated using the
\code{link{create_erd}} function. This object encapsulates the data frames
representing the entities and the relationships between these entities.}

\item{label_distance}{A numeric value that specifies the distance between the
edge labels and the lines connecting the nodes. The default value is 2.5.}

\item{label_angle}{A numeric value that specifies the angle at which the edge
labels are displayed. The default value is 45 degrees.}

\item{n}{The maximum number of rows in each table. The tables will add
columns to show all of the variables in the tables such that there are only
`n` rows.}
}
\value{
A \code{DiagrammeR} graph object representing the ERD.
}
\description{
The \code{\link{render_erd}} function graphically renders ERD using
\code{DiagrammeR}, incorporating pseudo-nodes to depict traditional ERD
notations such as cardinality. This function uses edge attributes to append
text labels near the end of the edge lines. Specifically, the
\code{DiagrammeR} label attribute is leveraged to include text labels at the
ends of the edges, which effectively convey the intended cardinality and
relationship information. This function constructs edge labels from two
strings representing the left and right relationship attributes.
}
\details{
This function is responsible for graphically rendering an Entity-Relationship
Diagram (ERD) based on an object of class "ERD". This function leverages the
\code{DiagrammeR} package to generate a graph that visually represents both
the entities and the relationships contained within the ERD object. The
entities are represented as nodes, and the relationships as edges. The
\code{render_erd} function thus provides a robust mechanism for visually
representing an ERD based on structured data within the R environment. By
converting an ERD object into a graphical form, the function aids in a
clearer understanding and communication of complex data relationships.

Nodes: Each entity (i.e., data frame) is represented as a node. The node
label consists of the entity name and the attribute names within the entity.

Edges: Relationships between entities are represented as edges between the
corresponding nodes. Labels at the ends of the edges indicate the type and
cardinality of the relationship.

The \code{label_distance} and \code{label_angle} parameters control the
presentation of edge labels in the ERD to minimize overlap and improve
readability.

The function uses a for loop to iterate through the entities and
relationships, constructing the necessary \code{DiagrammeR} code to render
each element. The \code{nodesep} and \code{ranksep} parameters in the
\code{DiagrammeR} code control the node spacing in the rendered ERD, making
it easier to visualize complex ERDs.
}
\examples{


# Load Packages -----------------------------------------------------------

library(ERDbuilder)
library(dplyr)

# Define entities ---------------------------------------------------------

students_tbl <- data.frame(
  st_id = c("hu1", "de2", "lo3"),
  dep_id = c("water", "evil", "values"),
  student = c("Huey", "Dewey", "Louie"),
  email = c("hubert.duck", "dewfort.duck", "llewellyn.duck"),
  dob = c("04-15", "04-15", "04-15")
)

courses_tbl <- data.frame(
  crs_id = c("water101", "evil205", "water202"),
  fac_id = c("02do", "03pe", "04mi"),
  dep_id = c("water", "evil", "water"),
  course = c("Swimming", "Human-chasing", "Dives")
)

enrollment_tbl <- data.frame(
  crs_id = c("water101", "evil205", "evil205", "water202"),
  st_id = c("hu1", "hu1", "de2", "de2"),
  final_grade = c("B", "A", "A", "F")
)

department_tbl <- data.frame(
  dep_id = c("water", "evil", "values"),
  department = c("Water activities", "Evil procurement", "Good values")
)

faculty_tbl <- data.frame(
  faculty_name = c("Scrooge McDuck", "Donald", "Pete", "Mickey"),
  title = c("Emeritus", "Full", "Assistant", "Full"),
  fac_id = c("01sc", "02do", "03pe", "04mi"),
  dep_id = c("water", "water", "evil", "values")
)

head(students_tbl)
head(courses_tbl)
head(enrollment_tbl)
head(department_tbl)
head(faculty_tbl)

## Define relationships----------------------------------------
relationships <- list(
  courses = list(
    enrollment = list(crs_id = "crs_id", relationship = c("||", "|<")),
    department = list(dep_id = "dep_id", relationship = c(">|", "||")),
    faculty = list(fac_id = "fac_id", relationship = c(">0", "||"))
  ),
  enrollment = list(
    students = list(st_id = "st_id", relationship = c(">0", "||")
    )
  ),
  students = list(
    department = list(dep_id = "dep_id", relationship = c(">|", "||"))
  ),
  faculty = list(
    department = list(dep_id = "dep_id", relationship = c(">|", "||"))
  )
)

## Create ERD object
erd_object <-
  create_erd(
    list(
      students = students_tbl,
      courses = courses_tbl,
      enrollment = enrollment_tbl,
      department = department_tbl,
      faculty = faculty_tbl
    ),
    relationships)

## Render ERD -----------------------------------------------------------
render_erd(erd_object, label_distance = 0, label_angle = 15, n = 20)
}
