% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/maxlogLreg.R
\encoding{UTF-8}
\name{maxlogLreg}
\alias{maxlogLreg}
\title{Maximum Likelihood Estimation for parametric linear regression models}
\usage{
maxlogLreg(
  formulas,
  y_dist,
  support = NULL,
  data = NULL,
  subset = NULL,
  fixed = NULL,
  link = NULL,
  optimizer = "nlminb",
  start = NULL,
  lower = NULL,
  upper = NULL,
  inequalities = NULL,
  control = NULL,
  silent = FALSE,
  StdE_method = c("optim", "numDeriv"),
  ...
)
}
\arguments{
\item{formulas}{a list of formula objects. Each element must have an \code{~},
with the terms on the right separated by \code{+} operators. The response
variable on the left side is optional. Linear predictor of each parameter must
be specified with the name of the parameter followed by the suffix \code{'.fo'}.
See the examples below for further illustration.}

\item{y_dist}{a formula object that specifies the distribution of the response
variable. On the left side of \code{~} must be the response,
and in the right side must be the name o de probability
density/mass function. See the section \strong{Details} and the
examples below for further illustration.}

\item{support}{a list with the following entries:
\itemize{
\item \code{interval}: a two dimensional atomic vector indicating
the set of possible values of a random variable having the
distribution specified in \code{y_dist}.
\item \code{type}: character indicating if distribution has a
\code{discrete} or a \code{continous} random variable.
}}

\item{data}{an optional data frame containing the variables in the model.
If data is not specified, the variables are taken from the
environment from which \code{maxlogLreg} is called.}

\item{subset}{an optional vector specifying a subset of observations to be used
in the fitting process.}

\item{fixed}{a list with fixed/known parameters of distribution of interest.
Fixed parameters must be passed with its name and its value (known).}

\item{link}{a list with names of parameters to be linked, and names of the
link function object. For names of parameters, please visit
documentation of density/mass function. There are three link
functions available: \code{\link{log_link}}, \code{\link{logit_link}}
and \code{\link{NegInv_link}}. Take into account: the order
used in argument \code{over} corresponds to the order in argument
\code{link}.}

\item{optimizer}{a length-one character vector with the name of optimization
routine. \code{\link[stats:nlminb]{nlminb}}, \code{\link[stats:optim]{optim}} and
\code{\link[DEoptim:DEoptim]{DEoptim}} are available; \code{\link[stats:nlminb]{nlminb}}
is the default routine.}

\item{start}{a numeric vector with initial values for the parameters to be
estimated. Zero is the default value.}

\item{lower}{a numeric vector with lower bounds, with the same lenght of
argument \code{start} (for box-constrained optimization). \code{-Inf}
is the default value.}

\item{upper}{a numeric vector with upper bounds, with the same lenght of
argument \code{start} (for box-constrained optimization). \code{Inf}
is the default value.}

\item{inequalities}{a character vector with the inequality constrains for
the distribution parameters.}

\item{control}{control parameters of the optimization routine. Please, visit
documentation of selected optimizer for further information.}

\item{silent}{logical. If TRUE, warnings of \code{maxlogL} are suppressed.}

\item{StdE_method}{a length-one character vector with the routine for Hessian
matrix computation. The This is needed for standard error
estimation. The options available are \code{"optim"} and
\code{"numDeriv"}. For further information, visit
\code{\link[stats]{optim}} or \code{\link[numDeriv]{hessian}}.}

\item{...}{Further arguments to be supplied to the optimization routine.}
}
\value{
A list with class \code{maxlogL} containing the following
lists:
\item{fit}{A list with output information about estimation and method used.}
\item{inputs}{A list with all input arguments.}
\item{outputs}{A list with additional information. The most important outputs
are:
\itemize{
\item \code{npar}: number of parameters.
\item \code{n}: sample size
\item \code{Stde_method}: standard error computation method.
\item \code{b_lenght}: a list with the number of regression parameters.
\item \code{design_matrix}: a list with the \eqn{\mathbf{X}} matrix
for each parameter, the response values (called \code{y}) and
the censorship matrix (called \code{status}). See the Details
section for further information.
}
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

Function to compute maximum likelihood estimators (MLE) of regression parameters
of any distribution implemented in \code{R} with covariates (linear predictors).
}
\details{
\code{maxlogLreg} computes programmatically the log-likelihood
(log L) function corresponding for the following model:

\deqn{
  y_i \stackrel{iid.}{\sim} \mathcal{D}(\theta_{i1},\theta_{i2},\dots,
  \theta_{ij}, \dots, \theta_{ik})
}
\deqn{
  g(\boldsymbol{\theta}_{j}) = \boldsymbol{\eta}_{j} = \mathbf{X}_j^\top
  \boldsymbol{\beta}_j,
}

where,

\itemize{
\item \eqn{g_k(\cdot)} is the \eqn{k}-th link function.
\item \eqn{\boldsymbol{\eta}_{j}} is the value of the linear predictor for the
\eqn{j^{th}} for all the observations.
\item \eqn{\boldsymbol{\beta}_j = (\beta_{0j}, \beta_{1j},\dots,
        \beta_{(p_j-1)j})^\top} are the fixed effects vector, where \eqn{p_j}
is the number of parameters in linear predictor \eqn{j} and
\eqn{\mathbf{X}_j} is a known design matrix of order \eqn{n\times p_j}.
These terms are specified in \code{formulas} argument.
\item \eqn{\mathcal{D}} is the distribution specified in argument
\code{y_dist}.
}

Then, \code{maxlogLreg} maximizes the log L through
\code{\link[stats:optim]{optim}}, \code{\link[stats:nlminb]{nlminb}} or
\code{\link[DEoptim:DEoptim]{DEoptim}}. \code{maxlogLreg} generates an S3 obj.

Estimation with censorship can be handled with \code{Surv} objects
(see example 2). The output object stores the corresponding censorship matrix,
defined as \eqn{r_{il} = 1} if sample unit \eqn{i} has status \eqn{l}, or
\eqn{r_{il} = 0} in other case. \eqn{i=1,2,\dots,n} and \eqn{l=1,2,3}
(\eqn{l=1}: observation status, \eqn{l=2}: right censorship status,
\eqn{l=3}: left censorship status).
}
\note{
\itemize{
\item The following generic functions can be used with a \code{maxlogL}
object: \code{summary, print, logLik, AIC}.
\item Noncentrality parameters must be named as \code{ncp} in the
distribution.
}
}
\examples{
library(EstimationTools)

#--------------------------------------------------------------------------------
# Example 1: Estimation in simulated normal distribution
n <- 1000
x <- runif(n = n, -5, 6)
y <- rnorm(n = n, mean = -2 + 3 * x, sd = exp(1 + 0.3* x))
norm_data <- data.frame(y = y, x = x)

# It does not matter the order of distribution parameters
formulas <- list(sd.fo = ~ x, mean.fo = ~ x)
support <- list(interval = c(-Inf, Inf), type = 'continuous')

norm_mod <- maxlogLreg(formulas, y_dist = y ~ dnorm, support = support,
                       data = norm_data,
                       link = list(over = "sd", fun = "log_link"))
summary(norm_mod)


#--------------------------------------------------------------------------------
# Example 2: Fitting with censorship
# (data from https://www.itl.nist.gov/div898/handbook/apr/section4/apr413.htm)

failures <- c(55, 187, 216, 240, 244, 335, 361, 373, 375, 386)
fails <- c(failures, rep(500, 10))
status <- c(rep(1, length(failures)), rep(0, 10))
Wei_data <- data.frame(fails = fails, status = status)

# Formulas with linear predictors
formulas <- list(scale.fo=~1, shape.fo=~1)
support <- list(interval = c(0, Inf), type = 'continuous')

# Bounds for optimization. Upper bound set with default values (Inf)
start <- list(
  scale = list(Intercept = 100),
  shape = list(Intercept = 10)
)
lower <- list(
  scale = list(Intercept = 0),
  shape = list(Intercept = 0)
)

mod_weibull <- maxlogLreg(formulas, y_dist = Surv(fails, status) ~ dweibull,
                          support = c(0, Inf), start = start,
                          lower = lower, data = Wei_data)
summary(mod_weibull)


#--------------------------------------------------------------------------------

}
\references{
\insertRef{Nelder1965}{EstimationTools}

\insertRef{Fox1978}{EstimationTools}

\insertRef{Nash1979}{EstimationTools}

\insertRef{Dennis1981}{EstimationTools}
}
\seealso{
\code{\link{summary.maxlogL}},
\code{\link[stats:optim]{optim}},
\code{\link[stats:nlminb]{nlminb}},
\code{\link[DEoptim:DEoptim]{DEoptim}},
\code{\link[DEoptim:DEoptim.control]{DEoptim.control}},
\code{\link{maxlogL}},
\code{\link{bootstrap_maxlogL}}

Other maxlogL: 
\code{\link{cum_hazard.maxlogL}()},
\code{\link{expected_value.maxlogL}()},
\code{\link{maxlogL}()}
}
\author{
Jaime Mosquera Gutiérrez, \email{jmosquerag@unal.edu.co}
}
\concept{maxlogL}
