reestimate <- function(x, wt, obsdist, eta, J, dwelldist, dwellpar, shift=FALSE) {

  # Helper function to normalize probabilities with smoothing
  normalize_probs <- function(col, eps) {
    (col + eps) / (sum(col) + eps * length(col))
  }

  ## Observation distribution parameter updating

  # Poisson distribution
  if(obsdist == "pois") {
    lambda <- numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      lambda[i] <- sum(x * wt[,i]) / sum(wt[,i])  # Weighted mean
    }
    obspar <- list(lambda = exp(log(lambda)))
  }

  # Normal distribution
  if(obsdist == "norm") {
    mean <- sd <- numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      valid_idx <- !is.na(x)
      data_subset <- data.frame(x[valid_idx])
      weights_subset <- wt[valid_idx,i]
      stats <- cov.wt(data_subset, weights_subset)  # Weighted moments
      mean[i] <- stats$center
      sd[i] <- sqrt(stats$cov)
    }
    obspar <- list(mean = mean, sd = sd)
  }

  # Weibull distribution
  if(obsdist == "weibull") {
    shape <- scale <-numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      valid_idx <- !is.na(x)
      data_subset <- x[valid_idx]
      weights_subset <- wt[valid_idx, i]

      # Weighted negative log-likelihood for Weibull
      log_likelihood <- function(params) {
        k <- params[1]  # shape
        lambda <- params[2]  # scale
        ll <- sum(weights_subset * (log(k) - k * log(lambda) +
                                      (k - 1) * log(data_subset) -
                                      (data_subset / lambda)^k))
        return(-ll)
      }

      init_params <- c(1, mean(data_subset))
      result <- optim(par = init_params, fn = log_likelihood,
                      method = "L-BFGS-B", lower = c(0.1, 0.1))

      shape[i] <- result$par[1]
      scale[i] <- result$par[2]
    }
    obspar <- list(shape = shape, scale = scale)
  }

  # Zero-inflated Poisson distribution
  if(obsdist == "zip") {
    pi <- lambda <- numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      valid_idx <- !is.na(x)
      data_subset <- x[valid_idx]
      weights_subset <- wt[valid_idx, i]

      # Estimate zero-inflation probability and Poisson rate
      w_zero <- sum(weights_subset[data_subset == 0])
      w_nonzero <- sum(weights_subset[data_subset > 0])
      total_weight <- w_zero + w_nonzero

      pi[i] <- w_zero / total_weight  # Zero-inflation probability
      lambda[i] <- sum(data_subset[data_subset > 0] *
                         weights_subset[data_subset > 0]) /
        sum(weights_subset[data_subset > 0])  # Poisson rate for non-zeros
    }
    obspar <- list(pi = pi, lambda = lambda)
  }

  # Negative binomial distribution
  if(obsdist == "nbinom") {
    size <- mu <- numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      valid_idx <- !is.na(x)
      data_subset <- x[valid_idx]
      weights_subset <- wt[valid_idx, i]

      # Weighted negative log-likelihood for negative binomial
      log_likelihood <- function(params) {
        s <- params[1]  # size parameter
        m <- params[2]  # mean parameter
        ll <- sum(weights_subset * (lgamma(data_subset + s) -
                                      lgamma(s) - lgamma(data_subset + 1) +
                                      s * log(s/(s + m)) +
                                      data_subset * log(m/(s + m))))
        return(-ll)
      }

      init_params <- c(1, mean(data_subset))
      result <- optim(par = init_params, fn = log_likelihood,
                      method = "L-BFGS-B", lower = c(0.1, 0.1))

      size[i] <- result$par[1]
      mu[i] <- result$par[2]
    }
    obspar <- list(size = size, mu = mu)
  }

  # Zero-inflated negative binomial distribution
  if(obsdist == "zinb") {
    size <- mu <- pi <- numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      valid_idx <- !is.na(x)
      data_subset <- x[valid_idx]
      weights_subset <- wt[valid_idx, i]

      w_zero <- weights_subset[data_subset == 0]
      w_nonzero <- weights_subset[data_subset > 0]

      pi[i] <- sum(w_zero) / (sum(w_zero) + sum(w_nonzero))  # Zero-inflation probability

      # Fit negative binomial to non-zero data
      log_likelihood <- function(params) {
        s <- params[1]
        m <- params[2]
        nonzero_data <- data_subset[data_subset > 0]
        nonzero_weights <- weights_subset[data_subset > 0]
        ll <- sum(nonzero_weights * (lgamma(nonzero_data + s) -
                                       lgamma(s) - lgamma(nonzero_data + 1) +
                                       s * log(s/(s + m)) +
                                       nonzero_data * log(m/(s + m))))
        return(-ll)
      }

      init_params <- c(1, mean(data_subset[data_subset > 0]))
      result <- optim(par = init_params, fn = log_likelihood,
                      method = "L-BFGS-B", lower = c(0.1, 0.1))

      size[i] <- result$par[1]
      mu[i] <- result$par[2]
    }
    obspar <- list(size = size, mu = mu, pi = pi)
  }

  # Exponential distribution
  if(obsdist == "exp") {
    rate <- numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      rate[i] <- sum(wt[, i]) / sum(x * wt[, i])  # Weighted rate parameter
    }
    obspar <- list(rate = rate)
  }

  # Gamma distribution
  if(obsdist == "gamma") {
    shape <- rate <- numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      valid_idx <- !is.na(x)
      data_subset <- x[valid_idx]
      weights_subset <- wt[valid_idx, i]

      # Weighted negative log-likelihood for gamma distribution
      log_likelihood <- function(params) {
        alpha <- params[1]  # shape parameter
        beta <- params[2]   # rate parameter
        ll <- sum(weights_subset * ((alpha - 1) * log(data_subset) - beta * data_subset + alpha * log(beta) - lgamma(alpha)))
        return(-ll)
      }

      init_params <- c(1, 1 / mean(data_subset))  # Shape and rate initialization
      result <- optim(par = init_params, fn = log_likelihood, method = "L-BFGS-B", lower = c(0.1, 0.1))

      shape[i] <- result$par[1]
      rate[i] <- result$par[2]
    }
    obspar <- list(shape = shape, rate = rate)
  }

  # Log-normal distribution
  if(obsdist == "lnorm") {
    meanlog <- sdlog <- numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      valid_idx <- !is.na(x)
      data_subset <- log(x[valid_idx])  # Log-transform data
      weights_subset <- wt[valid_idx, i]
      stats <- cov.wt(data.frame(data_subset), weights_subset)  # Weighted moments of log data

      meanlog[i] <- stats$center
      sdlog[i] <- sqrt(stats$cov)
    }
    obspar <- list(meanlog = meanlog, sdlog = sdlog)
  }

  # Generalized extreme value distribution
  if(obsdist == "gev") {
    loc <- scale <- shape <- numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      valid_idx <- !is.na(x)
      data_subset <- x[valid_idx]
      weights_subset <- wt[valid_idx, i]

      # Handle degenerate cases
      if(length(data_subset) == 0 || sum(weights_subset) < 1e-10) {
        loc[i] <- mean(x, na.rm = TRUE)
        scale[i] <- sd(x, na.rm = TRUE)
        shape[i] <- 0
        next
      }

      # Try using evd package if available
      if(requireNamespace("evd", quietly = TRUE)) {
        tryCatch({
          fit <- evd::fgev(data_subset, weights = weights_subset)
          loc[i] <- fit$estimate[1]
          scale[i] <- fit$estimate[2]
          shape[i] <- fit$estimate[3]
          next
        }, error = function(e) {})
      }

      # Custom GEV likelihood optimization
      log_likelihood <- function(params) {
        mu <- params[1]     # location
        sigma <- exp(params[2])  # scale (log-parameterized)
        xi <- params[3]     # shape

        z <- (data_subset - mu) / sigma

        # Handle Gumbel case (xi ≈ 0)
        if(abs(xi) < 1e-6) {
          z_safe <- pmin(pmax(z, -50), 50)
          ll_values <- -log(sigma) - z_safe - exp(-z_safe)
        } else {
          term <- 1 + xi * z

          ll_values <- rep(-1e10, length(z))

          # Only compute likelihood for valid support
          valid_idx <- term > 0
          if(sum(valid_idx) < length(z) * 0.1) {
            return(1e10)
          }

          ll_values[valid_idx] <- -log(sigma) -
            (1 + 1/xi) * log(term[valid_idx]) -
            term[valid_idx]^(-1/xi)
        }

        ll_values[is.nan(ll_values) | is.infinite(ll_values)] <- -1e10

        ll <- sum(weights_subset * ll_values, na.rm = TRUE)

        if(is.nan(ll) || is.infinite(ll)) return(1e10)

        return(-ll)
      }

      # Initialize parameters
      init_mu <- weighted.mean(data_subset, weights_subset)

      init_sigma <- mad(data_subset, center = init_mu)
      if(is.na(init_sigma) || init_sigma < 1e-3) {
        init_sigma <- sqrt(weighted.mean((data_subset - init_mu)^2, weights_subset))
      }
      init_sigma <- max(init_sigma, 1e-3)

      # Try multiple optimization strategies
      best_result <- NULL
      best_value <- Inf

      for(xi_start in c(0, -0.1, 0.1, -0.2, 0.2, -0.3, 0.3)) {
        for(method in c("BFGS", "Nelder-Mead")) {
          tryCatch({
            control_params <- if(method == "BFGS") {
              list(maxit = 300, reltol = 1e-6)
            } else {
              list(maxit = 500, reltol = 1e-6)
            }

            result <- optim(
              par = c(init_mu, log(init_sigma), xi_start),
              fn = log_likelihood,
              method = method,
              control = control_params
            )

            if(!is.na(result$value) && result$value < best_value) {
              best_result <- result
              best_value <- result$value
            }
          }, error = function(e) {})
        }

        if(!is.null(best_result) && best_value < 1e5) break
      }

      # Fallback to constrained optimization
      if(is.null(best_result) || best_value > 1e5) {
        tryCatch({
          result <- optim(
            par = c(init_mu, log(init_sigma), 0),
            fn = log_likelihood,
            method = "L-BFGS-B",
            lower = c(-Inf, log(1e-5), -Inf),
            upper = c(Inf, Inf, Inf),
            control = list(maxit = 300, factr = 1e7)
          )

          if(!is.na(result$value) && (is.null(best_result) || result$value < best_value)) {
            best_result <- result
            best_value <- result$value
          }
        }, error = function(e) {})
      }

      # Extract final parameters or use fallback values
      if(is.null(best_result)) {
        loc[i] <- init_mu
        scale[i] <- init_sigma
        shape[i] <- 0
      } else {
        loc[i] <- best_result$par[1]
        scale[i] <- exp(best_result$par[2])
        shape[i] <- best_result$par[3]
      }
    }
    obspar <- list(loc = loc, scale = scale, shape = shape)
  }

  # Zero-inflated Normal distribution
  if(obsdist == "ZInormal") {
    mean <- sd <- pi <- numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      valid_idx <- !is.na(x)
      data_subset <- x[valid_idx]
      weights_subset <- wt[valid_idx, i]

      # Estimate zero-inflation probability
      w_zero <- sum(weights_subset[data_subset == 0])
      w_nonzero <- sum(weights_subset[data_subset != 0])
      total_weight <- w_zero + w_nonzero

      pi[i] <- w_zero / total_weight  # Zero-inflation probability

      # Estimate normal parameters for non-zero data
      if(sum(data_subset != 0) > 0) {
        nonzero_data <- data_subset[data_subset != 0]
        nonzero_weights <- weights_subset[data_subset != 0]

        # Weighted moments for non-zero data
        stats <- cov.wt(data.frame(nonzero_data), nonzero_weights)
        mean[i] <- stats$center
        sd[i] <- sqrt(stats$cov)
      } else {
        mean[i] <- 0
        sd[i] <- 1
      }
    }
    obspar <- list(mean = mean, sd = sd, pi = pi)
  }

  # Zero-inflated Gamma distribution
  if(obsdist == "ZIgamma") {
    shape <- rate <- pi <- numeric(dim(wt)[2])
    for(i in seq_len(dim(wt)[2])) {
      valid_idx <- !is.na(x)
      data_subset <- x[valid_idx]
      weights_subset <- wt[valid_idx, i]

      # Estimate zero-inflation probability
      w_zero <- sum(weights_subset[data_subset == 0])
      w_nonzero <- sum(weights_subset[data_subset > 0])
      total_weight <- w_zero + w_nonzero

      pi[i] <- w_zero / total_weight  # Zero-inflation probability

      # Estimate gamma parameters for non-zero data
      if(sum(data_subset > 0) > 0) {
        nonzero_data <- data_subset[data_subset > 0]
        nonzero_weights <- weights_subset[data_subset > 0]

        # Weighted negative log-likelihood for gamma distribution on non-zero data
        log_likelihood <- function(params) {
          alpha <- params[1]  # shape parameter
          beta <- params[2]   # rate parameter
          ll <- sum(nonzero_weights * ((alpha - 1) * log(nonzero_data) -
                                         beta * nonzero_data + alpha * log(beta) - lgamma(alpha)))
          return(-ll)
        }

        init_params <- c(1, 1 / mean(nonzero_data))  # Shape and rate initialization
        result <- optim(par = init_params, fn = log_likelihood,
                        method = "L-BFGS-B", lower = c(0.1, 0.1))

        shape[i] <- result$par[1]
        rate[i] <- result$par[2]
      } else {
        shape[i] <- 1
        rate[i] <- 1
      }
    }
    obspar <- list(shape = shape, rate = rate, pi = pi)
  }


  ## Dwelltime distribution parameter updating

  epsilon <- 1e-20
  eta_matrix <- matrix(eta, ncol = J)
  M <- length(eta_matrix[,1])

  # Poisson dwell-time distribution
  if(dwelldist == "pois") {
    dwellprobs <- apply(eta_matrix, 2, function(col) normalize_probs(col, epsilon))
    lambda <- numeric(J)

    # Handle distribution shifting
    if(shift == FALSE) {
      shift <- rep(1, J)
    } else {
      shift <- numeric(J)
      for(i in seq_len(J)) {
        probs <- dwellprobs[,i]
        start_idx <- which.max(probs > epsilon)
        end_idx <- max(which(probs > epsilon))

        # Find optimal shift by maximizing likelihood
        shift_calc <- function(s) {
          idx_range <- start_idx:end_idx
          shifted_vals <- idx_range - s
          lambda_temp <- sum(shifted_vals * probs[idx_range])
          sum(log(dpois(idx_range - s, lambda_temp)) * probs[idx_range])
        }
        shift[i] <- which.max(sapply(seq_len(start_idx), shift_calc))
      }
    }

    # Estimate Poisson parameters for each state
    for(i in seq_len(J)) {
      probs <- dwellprobs[,i]
      end_idx <- max(which(probs > epsilon))
      range_vals <- shift[i]:end_idx
      lambda[i] <- sum((range_vals - shift[i]) * probs[range_vals])  # Expected dwell time
      dwellprobs[,i] <- dpois(seq_len(M) - shift[i], lambda[i])
    }

    dwellpar <- list(lambda = lambda, shift = shift)

    # Negative binomial dwell-time distribution
  } else if(dwelldist == "nbinom") {
    dwellprobs <- apply(eta_matrix, 2, function(col) normalize_probs(col, epsilon))
    size <- mu <- numeric(J)

    # Handle distribution shifting
    if(shift == FALSE) {
      shift <- rep(1, J)
    } else {
      shift <- numeric(J)
      for(i in seq_len(J)) {
        probs <- dwellprobs[,i]
        start_idx <- which.max(probs > epsilon)
        end_idx <- max(which(probs > epsilon))

        # Find optimal shift by maximizing likelihood
        shift_calc <- function(s) {
          idx_range <- start_idx:end_idx
          shifted_vals <- idx_range - s
          mean_val <- sum(shifted_vals * probs[idx_range])
          var_val <- sum((shifted_vals - mean_val)^2 * probs[idx_range])
          size_temp <- mean_val^2 / (var_val - mean_val)
          size_temp <- max(0.1, size_temp)
          mu_temp <- mean_val
          sum(log(dnbinom(idx_range - s, size = size_temp, mu = mu_temp)) * probs[idx_range])
        }
        shift[i] <- which.max(sapply(seq_len(start_idx), shift_calc))
      }
    }

    # Estimate negative binomial parameters for each state
    for(i in seq_len(J)) {
      probs <- dwellprobs[,i]
      end_idx <- max(which(probs > epsilon))
      range_vals <- shift[i]:end_idx
      shifted_vals <- range_vals - shift[i]

      # Optimize negative binomial parameters
      nll <- function(params) {
        size_temp <- exp(params[1])
        mu_temp <- exp(params[2])
        -sum(probs[range_vals] * dnbinom(
          shifted_vals,
          size = size_temp,
          mu = mu_temp,
          log = TRUE
        ))
      }

      mean_val <- sum(shifted_vals * probs[range_vals])
      var_val <- sum((shifted_vals - mean_val)^2 * probs[range_vals])
      init_size <- mean_val^2 / (var_val - mean_val)
      result <- optim(
        par = c(log(max(0.1, init_size)), log(mean_val)),
        fn = nll,
        method = "BFGS",
        control = list(maxit = 1000)
      )
      size[i] <- exp(result$par[1])
      mu[i] <- exp(result$par[2])
      dwellprobs[,i] <- dnbinom(
        seq_len(M) - shift[i],
        size = size[i],
        mu = mu[i]
      )
    }

    dwellpar <- list(size = size, mu = mu, shift = shift)

    # Beta-binomial dwell-time distribution
  } else if(dwelldist == "betabinom") {
    eta_matrix <- matrix(eta, ncol = J)
    dwellprobs <- apply(eta_matrix, 2, function(col) (col + epsilon) / (sum(col) + epsilon * length(col)))
    alpha <- beta <- size <- numeric(J)
    M <- length(eta_matrix[,1])

    # Handle distribution shifting
    if(shift == FALSE) {
      shift <- rep(1, J)
    } else {
      shift <- numeric(J)
      for(i in seq_len(J)) {
        probs <- dwellprobs[,i]
        start_idx <- which.max(probs > epsilon)
        end_idx <- max(which(probs > epsilon))

        # Find optimal shift by maximizing likelihood
        shift_calc <- function(s) {
          idx_range <- start_idx:end_idx
          shifted_vals <- idx_range - s

          # Beta-binomial likelihood function
          log_likelihood <- function(params) {
            alpha_temp <- exp(params[1])
            beta_temp <- exp(params[2])
            size_temp <- max(max(shifted_vals), round(exp(params[3])))
            ll <- sum(probs[idx_range] * dbetabinom(shifted_vals,
                                                    size = size_temp,
                                                    alpha = alpha_temp,
                                                    beta = beta_temp,
                                                    log = TRUE))
            if(is.nan(ll) || is.infinite(ll)) return(1e10)
            return(-ll)
          }

          # Method of moments initialization
          mean_val <- weighted.mean(shifted_vals, probs[idx_range])
          var_val <- weighted.mean((shifted_vals - mean_val)^2, probs[idx_range])
          n <- max(shifted_vals)
          p <- mean_val / n
          v <- var_val / (n^2)

          # Add small regularization to prevent division by zero
          v <- pmax(v, p * (1 - p) / n + 1e-6)

          common_term <- p * (1 - p) / v - 1
          common_term <- pmax(common_term, 0.1)  # Prevent negative or too small values

          init_alpha <- p * common_term
          init_beta <- (1 - p) * common_term

          # Bound initial values to prevent blowup
          init_alpha <- pmax(pmin(init_alpha, 50), 0.1)
          init_beta <- pmax(pmin(init_beta, 50), 0.1)

          init_params <- log(c(init_alpha, init_beta, max(shifted_vals)))

          tryCatch({
            result <- optim(par = init_params,
                            fn = log_likelihood,
                            method = "BFGS",
                            control = list(maxit = 100))  # Reduced iterations
            return(-result$value)
          }, error = function(e) return(-Inf))
        }
        shift[i] <- which.max(sapply(seq_len(start_idx), shift_calc))
      }
    }

    # Estimate beta-binomial parameters for each state
    for(i in seq_len(J)) {
      probs <- dwellprobs[,i]
      end_idx <- max(which(probs > epsilon))
      range_vals <- shift[i]:end_idx

      # Beta-binomial likelihood optimization
      log_likelihood <- function(params) {
        alpha_temp <- exp(params[1])
        beta_temp <- exp(params[2])
        size_temp <- max(max(range_vals - shift[i]), round(exp(params[3])))
        ll <- sum(probs[range_vals] * dbetabinom(range_vals - shift[i],
                                                 size = size_temp,
                                                 alpha = alpha_temp,
                                                 beta = beta_temp,
                                                 log = TRUE))
        if(is.nan(ll) || is.infinite(ll)) return(1e10)
        return(-ll)
      }

      # Method of moments initialization with regularization
      mean_val <- weighted.mean(range_vals - shift[i], probs[range_vals])
      var_val <- weighted.mean((range_vals - shift[i] - mean_val)^2, probs[range_vals])
      n <- max(range_vals - shift[i])
      p <- mean_val / n
      v <- var_val / (n^2)

      # Key fix: Add regularization to prevent parameter blowup
      v <- pmax(v, p * (1 - p) / n + 1e-4)  # Ensure minimum variance

      common_term <- p * (1 - p) / v - 1
      common_term <- pmax(common_term, 1)  # Minimum concentration
      common_term <- pmin(common_term, 100)  # Maximum concentration to prevent blowup

      init_alpha <- p * common_term
      init_beta <- (1 - p) * common_term

      # Additional bounds on initial parameters
      init_alpha <- pmax(pmin(init_alpha, 20), 0.5)
      init_beta <- pmax(pmin(init_beta, 20), 0.5)

      init_params <- log(c(init_alpha, init_beta, max(range_vals - shift[i])))

      # Single optimization attempt with bounds
      result <- tryCatch({
        optim(par = init_params,
              fn = log_likelihood,
              method = "L-BFGS-B",
              lower = log(c(0.1, 0.1, max(range_vals - shift[i]))),
              upper = log(c(20, 20, 2 * max(range_vals - shift[i]))),
              control = list(maxit = 200))
      }, error = function(e) NULL)

      # Extract parameters with fallback
      if(!is.null(result) && is.finite(result$value)) {
        alpha[i] <- exp(result$par[1])
        beta[i] <- exp(result$par[2])
        size[i] <- max(max(range_vals - shift[i]), round(exp(result$par[3])))
      } else {
        # Fallback to method of moments
        alpha[i] <- init_alpha
        beta[i] <- init_beta
        size[i] <- max(range_vals - shift[i])
      }

      # Final parameter bounds to ensure stability
      alpha[i] <- pmax(pmin(alpha[i], 20), 0.1)
      beta[i] <- pmax(pmin(beta[i], 20), 0.1)

      dwellprobs[,i] <- dbetabinom(seq_len(M) - shift[i],
                                   size = size[i],
                                   alpha = alpha[i],
                                   beta = beta[i])
    }

    dwellpar <- list(alpha = alpha, beta = beta, size = size, shift = shift)

    # Geometric dwell-time distribution
  } else if (dwelldist == "geom") {
    eta_matrix <- matrix(eta, ncol = J)
    dwellprobs <- apply(eta_matrix, 2, function(col) normalize_probs(col, epsilon))
    prob <- numeric(J)

    # Estimate geometric parameters for each state
    for (i in seq_len(J)) {
      probs <- dwellprobs[, i]
      support <- seq_len(M)
      mean_val <- sum(support * probs)  # Expected dwell time
      prob[i] <- 1 / mean_val  # Success probability
      dwellprobs[, i] <- dgeom(seq_len(M) - 1, prob[i])  # Geometric pmf (0-indexed)
    }

    dwellpar <- list(prob = prob)
  }

  # Calculate survival function for each state
  surv <- matrix(0, nrow = M, ncol = J)
  for(j in seq_len(J)) {
    temp_vec <- dwellprobs[,j]
    for(i in seq_len(M)) {
      surv[i,j] <- sum(temp_vec[i:M])  # P(dwell time >= i)
    }
  }

  # Return updated parameters and probabilities
  return(list(obspar = obspar, dwellpar = dwellpar,
              dwellprobs = dwellprobs, surv = surv))
}

