#'@title Integrated Variance Correlation
#'@description This function is used to calculate the integrated variance correlation between two random variables or between a random variable and a multivariate random variable
#'@param y is a numeric vector
#'@param x is a numeric vector or a data matrix
#'@param K is the number of quantile levels
#'@param NN is the number of B spline basis, default is 3
#'@param type is an indicator for measuring linear or nonlinear correlation, "linear" represents linear correlation and "nonlinear" represents linear or nonlinear correlation using B splines
#'@importFrom splines bs
#'@importFrom quantreg rq
#'@return The value of the corresponding sample statistic
#'@examples
#'# linear model
#'n=100
#'x=rnorm(n)
#'y=3*x+rnorm(n)
#'
#'IVC(y,x,K=5,type="linear")
#'# nonlinear model
#'n=100
#'p=3
#'x=matrix(NA,nrow=n,ncol=p)
#'for(i in 1:p){
#'  x[,i]=rnorm(n)
#'}
#'y=cos(x[,1]+x[,2])+x[,3]^2+rnorm(n)
#'IVC(y,x,K=5,type="nonlinear")
#'@export
IVC<-function(y,x,K,NN=3,type){
  new_env1 = new.env()
  new_env1$x = x
  new_env1$y = y
  n=length(new_env1$y)
  int_tau=sapply(1:K,function(i) (i/(K+1)))
  wei=rep(1/(K+1),K)
  xx=x
  if(type=="nonlinear")
  {
    if(ncol(as.matrix(xx))==1){
      B_x<-bs(xx,df=NN)
      B_x=as.matrix(B_x)
    }
    if(ncol(as.matrix(xx))>1){
      B_x=NULL
      for (pp in 1:ncol(xx)) {
        xB=x[,pp]
        B_xx=bs(xB,df=NN)
        B_x=cbind(B_x,B_xx)
      }
    }
    rec_int=c()
    for (l in 1:length(int_tau)) {
      fitted_con=rq(y~B_x,tau=int_tau[l])$fitted.values
      ind1=c()
      for (t in 1:n) {
        ind1[t]=((length(which(y<=fitted_con[t]))/n)-int_tau[l])^2
      }
      rec_int[l]=sum(ind1)/n
    }
    IVcor=6*sum(wei*rec_int)
  }
  if(type=="linear")
  {
    B_x<-xx
    B_x=as.matrix(B_x)
    rec_int=c()
    for (l in 1:length(int_tau)) {
      fitted_con=rq(y~B_x,tau=int_tau[l])$fitted.values
      ind1=c()
      for (t in 1:n) {
        ind1[t]=((length(which(y<=fitted_con[t]))/n)-int_tau[l])^2
      }
      rec_int[l]=sum(ind1)/n
    }
    IVcor=6*sum(wei*rec_int)
  }
  return(IVcor)
}

#'@title Integrated Variance Correlation with Local Linear Estimation
#'@description This function is used to calculate the integrated variance correlation between two random variables with local linear estimation
#'@param y is a numeric vector
#'@param x is a numeric vector 
#'@param K is the number of quantile levels
#'@importFrom BwQuant bwRT
#'@importFrom quantdr llqr
#'@return The value of the corresponding sample statistic
#'@examples
#'n=100
#'x=rnorm(n)
#'y=exp(x)+rnorm(n)
#'
#'IVCLLQ(y,x,K=4)
#'@export
IVCLLQ<-function(y,x,K){
  new_env2 = new.env()
  new_env2$x = x
  new_env2$y = y
  n=length(new_env2$y)
  int_tau=sapply(1:K,function(i) (i/(K+1)))
  wei=rep(1/(K+1),K)
  xx=x
  rec_int=c()
  for (l in 1:length(int_tau)) {
    hh=try(bwRT(xx,y, int_tau[l]),silent = T)
    fitted_con=try(quantdr::llqr(xx,y,tau=int_tau[l],h=hh)$ll_est,silent = T)
    ind1=c()
    for (t in 1:n) {
      ind1[t]=((length(which(y<=fitted_con[t]))/n)-int_tau[l])^2
    }
    rec_int[l]=sum(ind1)/n
  }
  IVcor=6*sum(wei*rec_int)
  return(IVcor)
}

#'@title Integrated Variance Correlation with Discrete Response Variable
#'@description This function is used to calculate the integrated variance correlation between a discrete response variable and a continuous random variable
#'@param y is the categorical response vector
#'@param x is a numeric vector 
#'@param K is the number of quantile levels
#'@importFrom stats quantile
#'@return The value of the corresponding sample statistic
#'@examples
#'n=100
#'y=sample(rep(1:3), n, replace = TRUE, prob = c(1/3,1/3,1/3))
#'x=c()
#'for(i in 1:n){
#'  x[i]=rnorm(1,mean=2*y[i],sd=1)
#'}
#'
#'IVCCA(y,x,K=5)
#'@export
IVCCA<-function(y,x,K){
  new_env3 = new.env()
  new_env3$x = x
  new_env3$y = y
  n=length(new_env3$y)
  y_ca=unique(y)
  R=length(y_ca)
  y_pr=sapply(1:R,function(i) length(which(y==y_ca[i]))/n)
  
  int_tau=sapply(1:K,function(i) (i/(K+1)))
  wei=rep((1/(K+1)),K)
  rec_int=c()
  for (rr in 1:R){
    x_rr=x[which(y==y_ca[rr])]
    ind1=c()
    for (l in 1:length(int_tau)){
      ind1[l]=((length(which(x<=quantile(x_rr,int_tau[l])))/n)-int_tau[l])^2
    }
    rec_int[rr]=sum(wei*ind1)
  }
  IVcor=6*sum(y_pr*rec_int)
  return(IVcor)
}


#'@title Integrated Variance Correlation for Interval Independence
#'@description This function is used to calculate the integrated variance correlation to measure interval independence
#'@param y is a numeric vector
#'@param x is a numeric vector or a data matrix
#'@param tau1 is the minimum quantile level
#'@param tau2 is the maximum quantile level
#'@param K is the number of quantile levels
#'@param NN is the number of B spline basis, default is 3
#'@param type is an indicator for measuring linear or nonlinear correlation, "linear" represents linear correlation and "nonlinear" represents linear or nonlinear correlation using B splines
#'@importFrom splines bs
#'@importFrom quantreg rq
#'@return The value of the corresponding sample statistic for interval independence
#'@examples
#'# linear model
#'require("mvtnorm")
#'n=100
#'p=3
#'pho1=0.5
#'mean_x=rep(0,p)
#'sigma_x=matrix(NA,nrow = p,ncol = p)
#'for (i in 1:p) {
#'  for (j in 1:p) {
#'    sigma_x[i,j]=pho1^(abs(i-j))
#'  }
#'}
#'x=rmvnorm(n, mean = mean_x, sigma = sigma_x,method = "chol")
#'y=2*(x[,1]+x[,2]+x[,3])+rnorm(n)
#'
#'IVC_Interval(y,x,K=5,tau1=0.4,tau2=0.6,type="linear")
#'# nonlinear model
#'n=100
#'x=runif(n,min=-2,max=2)
#'y=exp(x^2)*rnorm(n)
#'
#'IVC_Interval(y,x,K=5,tau1=0.4,tau2=0.6,type="nonlinear")
#'@export
IVC_Interval<-function(y,x,K,tau1,tau2,NN=3,type){
  new_env4 = new.env()
  new_env4$x = x
  new_env4$y = y
  n=length(new_env4$y)
  int_tau1=sapply(1:K,function(i) (i/(K+1)))
  int_tau=tau1+(tau2-tau1)*int_tau1
  wei=rep((tau2-tau1)*(1/(K+1)),K)
  xx=x
  if(type=="nonlinear")
  {
    if(ncol(as.matrix(xx))==1){
      B_x<-bs(xx,df=NN)
      B_x=as.matrix(B_x)
    }
    if(ncol(as.matrix(xx))>1){
      B_x=NULL
      for (pp in 1:ncol(xx)) {
        xB=x[,pp]
        B_xx=bs(xB,df=NN)
        B_x=cbind(B_x,B_xx)
      }
    }
    rec_int=c()
    for (l in 1:length(int_tau)) {
      fitted_con=rq(y~B_x,tau=int_tau[l])$fitted.values
      ind1=c()
      for (t in 1:n) {
        ind1[t]=((length(which(y<=fitted_con[t]))/n)-int_tau[l])^2
      }
      rec_int[l]=sum(ind1)/n
    }
    index1=((tau2^3-tau1^3)/3)-((tau2^2-tau1^2)/2)+(tau2-tau1)/3
    IVcor=index1^(-1)*sum(wei*rec_int)
  }
  if(type=="linear")
  {
    B_x<-xx
    B_x=as.matrix(B_x)
    rec_int=c()
    for (l in 1:length(int_tau)) {
      fitted_con=rq(y~B_x,tau=int_tau[l])$fitted.values
      ind1=c()
      for (t in 1:n) {
        ind1[t]=((length(which(y<=fitted_con[t]))/n)-int_tau[l])^2
      }
      rec_int[l]=sum(ind1)/n
    }
    index1=((tau2^3-tau1^3)/3)-((tau2^2-tau1^2)/2)+(tau2-tau1)/3
    IVcor=index1^(-1)*sum(wei*rec_int)
  }
  return(IVcor)
}

#'@title Integrated Variance Correlation Based Hypothesis Test
#'@description This function is used to test significance of linear or nonlinear correlation using integrated variance correlation
#'@param y is the response vector
#'@param x is a numeric vector or a data matrix
#'@param K is the number of quantile levels
#'@param num_per is the number of permutation times
#'@param NN is the number of B spline basis, default is 3
#'@param type is an indicator for measuring linear or nonlinear correlation, "linear" represents linear correlation and "nonlinear" represents linear or nonlinear correlation using B splines
#'@importFrom splines bs
#'@importFrom quantreg rq
#'@return The p-value of the corresponding hypothesis test
#'@examples
#'# linear model
#'n=100
#'x=rnorm(n)
#'y=rnorm(n)
#'
#'IVCT(y,x,K=5,num_per=20,type = "linear")
#'# nonlinear model
#'n=100
#'p=4
#'x=matrix(NA,nrow=n,ncol=p)
#'for(i in 1:p){
#'  x[,i]=runif(n,0,1)
#'}
#'y=3*ifelse(x[,1]>0.5,1,0)*x[,2]+3*cos(x[,3])^2*x[,1]+3*(x[,4]^2-1)*x[,1]+rnorm(n)
#'
#'IVCT(y,x,K=5,num_per=20,type = "nonlinear")
#'@export
IVCT<-function(y,x,K,num_per,NN=3,type){
  new_env5 = new.env()
  new_env5$x = x
  new_env5$y = y
  n=length(new_env5$y)
  int_tau=sapply(1:K,function(i) (i/(K+1)))
  wei=rep(1/(K+1),K)
  xx=x
  if(type=="nonlinear"){
    if(ncol(as.matrix(xx))==1){
      B_x<-bs(xx,df=NN)
      B_x=as.matrix(B_x)
    }
    if(ncol(as.matrix(xx))>1){
      B_x=NULL
      for (pp in 1:ncol(xx)) {
        xB=x[,pp]
        B_xx=bs(xB,df=NN)
        B_x=cbind(B_x,B_xx)
      }
    }
    rec_int=c()
    for (l in 1:length(int_tau)) {
      fitted_con=rq(y~B_x,tau=int_tau[l])$fitted.values
      ind1=c()
      for (t in 1:n) {
        ind1[t]=((length(which(y<=fitted_con[t]))/n)-int_tau[l])^2
      }
      rec_int[l]=sum(ind1)/n
    }
    IMVT=6*(n^(4/5))*sum(wei*rec_int)
    
    pr_imv=c()
    new_env5$pr_imv = pr_imv
    for (o in 1:num_per) {
      pr_sim=sample(1:n,n,replace =FALSE)
      y_pe=y[pr_sim]
      rec_int=c()
      for (l in 1:length(int_tau)) {
        fitted_con=rq(y_pe~B_x,tau=int_tau[l])$fitted.values
        ind1=c()
        for (t in 1:n) {
          ind1[t]=((length(which(y_pe<=fitted_con[t]))/n)-int_tau[l])^2
        }
        rec_int[l]=sum(ind1)/n
      }
      pr_imv[o]=6*(n^(4/5))*sum(wei*rec_int)
    }
    p.value=length(which(pr_imv>=IMVT))/num_per 
  }
  if(type=="linear"){
    B_x<-xx
    B_x=as.matrix(B_x)
    rec_int=c()
    for (l in 1:length(int_tau)) {
      fitted_con=rq(y~B_x,tau=int_tau[l])$fitted.values
      ind1=c()
      for (t in 1:n) {
        ind1[t]=((length(which(y<=fitted_con[t]))/n)-int_tau[l])^2
      }
      rec_int[l]=sum(ind1)/n
    }
    IMVT=6*n*sum(wei*rec_int)
    
    pr_imv=c()
    new_env5$pr_imv = pr_imv
    for (o in 1:num_per) {
      pr_sim=sample(1:n,n,replace =FALSE)
      y_pe=y[pr_sim]
      rec_int=c()
      for (l in 1:length(int_tau)) {
        fitted_con=rq(y_pe~B_x,tau=int_tau[l])$fitted.values
        ind1=c()
        for (t in 1:n) {
          ind1[t]=((length(which(y_pe<=fitted_con[t]))/n)-int_tau[l])^2
        }
        rec_int[l]=sum(ind1)/n
      }
      pr_imv[o]=6*n*sum(wei*rec_int)
    }
    p.value=length(which(pr_imv>=IMVT))/num_per 
  }
  return(p.value)
}

#'@title Critical Values for Integrated Variance Correlation Based Hypothesis Test
#'@description This function is used to calculate the critical values for integrated variance correlation test at significance level 0.1, 0.05 and 0.01
#'@param N is a integer as large as possible, default is 500
#'@param realizations is the the number of replication times for simulating the distribution under the null hypothesis
#'@importFrom stats rchisq quantile
#'@return The critical values at significance level 0.1, 0.05 and 0.01
#'@examples
#'IVC_crit(N=500,realizations=100)
#'@export
IVC_crit <- function(N = 500, realizations) {
  x <- numeric(realizations)
  hm <- sum(1 / (1:N)^2) 
  for (i in 1:realizations){
    xi <- 0
    for (j in 1:N){
      xi <- xi + rchisq(1, 1) / j^2
    }
    x[i] <- (6*xi) / pi^2 + (1 - ((6*hm) / pi^2))    
  }
  quantile(x, probs = c(0.9, 0.95, 0.99))
}

#'@title Integrated Variance Correlation Based Hypothesis Test for Discrete Response
#'@description This function is used to test independence between a categorical variable and a continuous variable using integrated variance correlation
#'@param y is a categorical response vector
#'@param x is a numeric vector
#'@param K is the number of quantile levels
#'@param num_per is the number of permutation times
#'@param type is an indicator for fixed number of categories or infinity number of categories, "fixed" represents number of categories is fixed, then a permutation test is used, "infinity" represents number of categories is infinite, then an asymptotic normal distribution is used to calculate p values
#'@importFrom stats pnorm quantile
#'@return The p-value of the corresponding hypothesis test
#'@examples
#'# small R
#'n=100
#'x=runif(n,0,1)
#'y=sample(rep(1:3), n, replace = TRUE, prob = c(1/3,1/3,1/3))
#'
#'IVCCAT(y,x,K=5,num_per=20,type = "fixed")
#'# large R
#'n=200
#'y=sample(rep(1:20), n, replace = TRUE, prob = rep(1/20,20))
#'mu_x=sample(c(1,2,3,4),20,replace = TRUE,prob = c(1/4,1/4,1/4,1/4))
#'x=c()
#'for (i in 1:n) {
#'  x[i]=2*mu_x[y[i]]+rcauchy(1)
#'}
#'
#'IVCCAT(y,x,K=10,type = "infinity")
#'@export
IVCCAT<-function(y,x,K,num_per,type){
  new_env6 = new.env()
  new_env6$x = x
  new_env6$y = y
  n=length(new_env6$y)
  int_tau=sapply(1:K,function(i) (i/(K+1)))
  wei=rep(1/(K+1),K)
  if(type=="fixed"){
    y_ca=unique(y)
    R=length(y_ca)
    y_pr=sapply(1:R,function(i) length(which(y==y_ca[i]))/n)
    
    int_tau=sapply(1:K,function(i) (i/(K+1)))
    wei=rep((1/(K+1)),K)
    rec_int=c()
    for (rr in 1:R){
      x_rr=x[which(y==y_ca[rr])]
      ind1=c()
      for (l in 1:length(int_tau)){
        ind1[l]=((length(which(x<=quantile(x_rr,int_tau[l])))/n)-int_tau[l])^2
      }
      rec_int[rr]=sum(wei*ind1)
    }
    IVCT=6*n*sum(y_pr*rec_int)
    
    pr_iv=c()
    new_env6$pr_iv = pr_iv
    for (o in 1:num_per) {
      pr_sim=sample(1:n,n,replace =FALSE)
      y_pe=y[pr_sim]
      y_pr2=sapply(1:R,function(i) length(which(y_pe==y_ca[i]))/n)
      rec_int=c()
      for (rr in 1:R){
        x_rr=x[which(y_pe==y_ca[rr])]
        ind1=c()
        for (l in 1:length(int_tau)){
          ind1[l]=((length(which(x<=quantile(x_rr,int_tau[l])))/n)-int_tau[l])^2
        }
        rec_int[rr]=sum(wei*ind1)
      }
      pr_iv[o]=6*n*sum(y_pr2*rec_int)
    }
    p.value=length(which(pr_iv>=IVCT))/num_per 
  }
  if(type=="infinity"){
    y_ca=unique(y)
    R=length(y_ca)
    y_pr=sapply(1:R,function(i) length(which(y==y_ca[i]))/n)
    
    int_tau=sapply(1:K,function(i) (i/(K+1)))
    wei=rep((1/(K+1)),K)
    rec_int=c()
    for (rr in 1:R){
      x_rr=x[which(y==y_ca[rr])]
      ind1=c()
      for (l in 1:length(int_tau)){
        ind1[l]=((length(which(x<=quantile(x_rr,int_tau[l])))/n)-int_tau[l])^2
      }
      rec_int[rr]=sum(wei*ind1)
    }
    IVCT=6*n*sum(y_pr*rec_int)
    p.value=1-pnorm(IVCT,mean=(R-1),sd=(4*(R-1))/5)
  }
  return(p.value)
}

#'@title Critical Values for Integrated Variance Correlation Based Hypothesis Test with Discrete Response
#'@description This function is used to calculate the critical values for integrated variance correlation test with discrete response at significance level 0.1, 0.05 and 0.01
#'@param R is the number of categories
#'@param N is a integer as large as possible, default is 500
#'@param realizations is the the number of replication times for simulating the distribution under the null hypothesis
#'@importFrom stats rchisq quantile
#'@return The critical values at significance level 0.1, 0.05 and 0.01
#'@examples
#'IVCCA_crit(R=5,N=500,realizations=100)
#'@export
IVCCA_crit <- function(R, N = 500, realizations) {
  x <- numeric(realizations)
  hm <- sum(1 / (1:N)^2) 
  for (i in 1:realizations){
    xi <- 0
    for (j in 1:N){
      xi <- xi + rchisq(1, R-1) / j^2
    }
    x[i] <- (6*xi) / pi^2 + (R-1) * (1 - ((6*hm) / pi^2))  
  }
  quantile(x, probs = c(0.9, 0.95, 0.99))
}


#'@title Integrated Variance Correlation Based Hypothesis Test with Local Linear Estimation
#'@description This function is used to test significance using integrated variance correlation with local linear estimation
#'@param y is a numeric vector
#'@param x is a numeric vector
#'@param K is the number of quantile levels
#'@param num_per is the number of permutation times
#'@importFrom BwQuant bwRT
#'@importFrom quantdr llqr
#'@return The p-value of the corresponding hypothesis test
#'@examples
#'n=100
#'x=runif(n,-1,1)
#'y=2*cos(2*x)+rnorm(n)
#'
#'\donttest{
#'IVCTLLQ(y,x,K=5,num_per=100)
#'}
#'@export
IVCTLLQ<-function(y,x,K,num_per){
  new_env7 = new.env()
  new_env7$x = x
  new_env7$y = y
  n=length(new_env7$y)
  int_tau=sapply(1:K,function(i) (i/(K+1)))
  wei=rep(1/(K+1),K)
  xx=x
  rec_int=c()
  for (l in 1:length(int_tau)) {
    hh=try(bwRT(xx,y, int_tau[l]),silent = T)
    fitted_con=try(quantdr::llqr(xx,y,tau=int_tau[l],h=hh)$ll_est,silent = T)
    ind1=c()
    for (t in 1:n) {
      ind1[t]=((length(which(y<=fitted_con[t]))/n)-int_tau[l])^2
    }
    rec_int[l]=sum(ind1)/n
  }
  IVCT=6*(n^(4/5))*sum(wei*rec_int)
  
  pr_iv=c()
  new_env7$pr_iv = pr_iv
  for (o in 1:num_per) {
    pr_sim=sample(1:n,n,replace =FALSE)
    y_pe=y[pr_sim]
    rec_int=c()
    for (l in 1:length(int_tau)) {
      hh=try(bwRT(xx,y_pe, int_tau[l]),silent = T)
      fitted_con=try(quantdr::llqr(xx,y_pe,tau=int_tau[l],h=hh)$ll_est,silent = T)
      ind1=c()
      for (t in 1:n) {
        ind1[t]=((length(which(y_pe<=fitted_con[t]))/n)-int_tau[l])^2
      }
      rec_int[l]=sum(ind1)/n
    }
    pr_iv[o]=6*(n^(4/5))*sum(wei*rec_int)
  }
  p.value=length(which(pr_iv>=IVCT))/num_per 
  return(p.value)
}




#'@title Integrated Variance Correlation Based Interval Independence Hypothesis Test
#'@description This function is used to test interval independence using integrated variance correlation
#'@param y is the response vector
#'@param x is a numeric vector or a data matrix
#'@param tau1 is the minimum quantile level
#'@param tau2 is the maximum quantile level
#'@param K is the number of quantile levels
#'@param num_per is the number of permutation times
#'@param NN is the number of B spline basis, default is 3
#'@param type is an indicator for measuring linear or nonlinear correlation, "linear" represents linear correlation and "nonlinear" represents linear or nonlinear correlation using B splines
#'@importFrom splines bs
#'@importFrom quantreg rq
#'@return The p-value of the corresponding hypothesis test
#'@examples
#'require("mvtnorm")
#'n=100
#'p=3
#'pho1=0.5
#'mean_x=rep(0,p)
#'sigma_x=matrix(NA,nrow = p,ncol = p)
#'for (i in 1:p) {
#'  for (j in 1:p) {
#'    sigma_x[i,j]=pho1^(abs(i-j))
#'  }
#'}
#'x=rmvnorm(n, mean = mean_x, sigma = sigma_x,method = "chol")
#'y=rnorm(n)
#'
#'IVCT_Interval(y,x,tau1=0.5,tau2=0.75,K=5,num_per=20,type = "linear")
#'
#'n=100
#'x_til=runif(n,min=-1,max=1)
#'y_til=rnorm(n)
#'epsilon=rnorm(n)
#'x=x_til+2*epsilon*ifelse(x_til<=-0.5&y_til<=-0.675,1,0)
#'y=y_til+2*epsilon*ifelse(x_til<=-0.5&y_til<=-0.675,1,0)
#'
#'IVCT_Interval(y,x,tau1=0.6,tau2=0.8,K=5,num_per=20,type = "nonlinear")
#'@export
IVCT_Interval<-function(y,x,tau1,tau2,K,num_per,NN=3,type){
  new_env8 = new.env()
  new_env8$x = x
  new_env8$y = y
  n=length(new_env8$y)
  int_tau1=sapply(1:K,function(i) (i/(K+1)))
  int_tau=tau1+(tau2-tau1)*int_tau1
  wei=rep((tau2-tau1)*(1/(K+1)),K)
  xx=x
  if(type=="nonlinear"){
    if(ncol(as.matrix(xx))==1){
      B_x<-bs(xx,df=NN)
      B_x=as.matrix(B_x)
    }
    if(ncol(as.matrix(xx))>1){
      B_x=NULL
      for (pp in 1:ncol(xx)) {
        xB=x[,pp]
        B_xx=bs(xB,df=NN)
        B_x=cbind(B_x,B_xx)
      }
    }
    rec_int=c()
    for (l in 1:length(int_tau)) {
      fitted_con=rq(y~B_x,tau=int_tau[l])$fitted.values
      ind1=c()
      for (t in 1:n) {
        ind1[t]=((length(which(y<=fitted_con[t]))/n)-int_tau[l])^2
      }
      rec_int[l]=sum(ind1)/n
    }
    IVCT=n*sum(wei*rec_int)
    
    pr_iv=c()
    new_env8$pr_iv = pr_iv
    for (o in 1:num_per) {
      pr_sim=sample(1:n,n,replace =FALSE)
      y_pe=y[pr_sim]
      rec_int=c()
      for (l in 1:length(int_tau)) {
        fitted_con=rq(y_pe~B_x,tau=int_tau[l])$fitted.values
        ind1=c()
        for (t in 1:n) {
          ind1[t]=((length(which(y_pe<=fitted_con[t]))/n)-int_tau[l])^2
        }
        rec_int[l]=sum(ind1)/n
      }
      pr_iv[o]=n*sum(wei*rec_int)
    }
    p.value=length(which(pr_iv>=IVCT))/num_per 
  }
  if(type=="linear"){
    B_x<-xx
    B_x=as.matrix(B_x)
    rec_int=c()
    for (l in 1:length(int_tau)) {
      fitted_con=rq(y~B_x,tau=int_tau[l])$fitted.values
      ind1=c()
      for (t in 1:n) {
        ind1[t]=((length(which(y<=fitted_con[t]))/n)-int_tau[l])^2
      }
      rec_int[l]=sum(ind1)/n
    }
    IVCT=n*sum(wei*rec_int)
    
    pr_iv=c()
    new_env8$pr_iv = pr_iv
    for (o in 1:num_per) {
      pr_sim=sample(1:n,n,replace =FALSE)
      y_pe=y[pr_sim]
      rec_int=c()
      for (l in 1:length(int_tau)) {
        fitted_con=rq(y_pe~B_x,tau=int_tau[l])$fitted.values
        ind1=c()
        for (t in 1:n) {
          ind1[t]=((length(which(y_pe<=fitted_con[t]))/n)-int_tau[l])^2
        }
        rec_int[l]=sum(ind1)/n
      }
      pr_iv[o]=n*sum(wei*rec_int)
    }
    p.value=length(which(pr_iv>=IVCT))/num_per 
  }
  return(p.value)
}