#' A helper sigmoid function.
#'
#' @param x A numeric value or vector.
#' @param a A numeric scaling factor (default is 1).
#' @return The sigmoid of x.
#' @export
sigmoid <- function( x, a = 1 ) { return( 1 / ( 1 + exp( -a * x ) ) ) }

#' @title Adaptive k-Nearest Neighbor Classifier
#' @description Implements the adaptive k-nearest neighbor (kK-NN) algorithm, which adjusts the neighborhood size for each sample based on a local curvature estimate. This method aims to improve classification performance, particularly in datasets with limited training samples.
#'
#' @param train A numeric matrix or data frame of the training data.
#' @param test A numeric matrix or data frame of the test data.
#' @param train_target A numeric or factor vector of class labels for the training data.
#' @param k The number of neighbors for the initial k-NN graph.
#' @param func The transformation function for curvatures ('log', 'cubic_root', or 'sigmoid').
#' @param quantize_method The quantization method to use: 'paper' (10 levels, default) or 'log2n' (k levels, where k = log2(n)).
#' @return A numeric or factor vector of predicted class labels for the test data.
#'
#' @examples
#' # Load necessary libraries
#' library(caret)
#'
#' # Load and prepare data (e.g., the Iris dataset)
#' data_iris <- iris
#' data <- as.matrix(data_iris[, 1:4])
#' target <- as.integer(data_iris$Species)
#'
#' # Standardize the data
#' data <- scale(data)
#'
#' # Split data into training and testing sets
#' set.seed(42)
#' train_index <- caret::createDataPartition(target, p = 0.5, list = FALSE)
#' train_data <- data[train_index, ]
#' test_data <- data[-train_index, ]
#' train_labels <- target[train_index]
#'
#' # Determine initial k value as log2(n)
#' initial_k <- round(log2(nrow(train_data)))
#' if (initial_k %% 2 == 0) {
#'    initial_k <- initial_k + 1
#' }
#'
#' # Run the kK-NN classifier using the default quantization method ('paper')
#' predictions_paper <- LCCkNN::kKNN(
#'    train = train_data,
#'    test = test_data,
#'    train_target = train_labels,
#'    k = initial_k
#' )
#'
#' # Run the kK-NN classifier using the 'log2n' quantization method
#' predictions_log2n <- LCCkNN::kKNN(
#'    train = train_data,
#'    test = test_data,
#'    train_target = train_labels,
#'    k = initial_k,
#'    quantize_method = 'log2n'
#' )
#'
#' # Evaluate the results (e.g., calculate balanced accuracy)
#' test_labels <- target[-train_index]
#' bal_acc_paper <- LCCkNN::balanced_accuracy_score(test_labels, predictions_paper)
#' bal_acc_log2n <- LCCkNN::balanced_accuracy_score(test_labels, predictions_log2n)
#' cat("Balanced Accuracy (paper Method):", bal_acc_paper, "\n")
#' cat("Balanced Accuracy (log2n Method):", bal_acc_log2n, "\n")
#'
#' @references Levada, A.L.M., Nielsen, F., Haddad, M.F.C. (2024). ADAPTIVE k-NEAREST NEIGHBOR CLASSIFIER BASED ON THE LOCAL ESTIMATION OF THE SHAPE OPERATOR. arXiv:2409.05084.
#' @export
kKNN <- function( train, test, train_target, k, func = "log", quantize_method = "paper" ) {
  n_train <- nrow( train )
  n_test <- nrow( test )
  labels <- numeric( n_test )

  # Determine the number of quantization levels
  if ( quantize_method == "paper" ) {
    n_levels <- 10
  } else if ( quantize_method == "log2n" ) {
    n_levels <- round( log2( n_train ) )
    if ( n_levels %% 2 == 0 ) {
      n_levels <- n_levels + 1
    }
  } else {
    stop( "Invalid quantization_method. Choose 'paper' or 'log2n'." )
  }

  # Compute curvature for training set
  curvaturas <- curvature_estimation( train, k )

  # Apply transformation
  if ( func == "log" ) {
    if ( any( curvaturas <= 0 ) ) {
      warning( "Log transformation applied to non-positive curvature values. Replacing with a small positive value." )
      curvaturas[ curvaturas <= 0 ] <- 1e-6
    }
    curvaturas <- log( curvaturas )
  } else if ( func == "cubic_root" ) {
    curvaturas <- sign( curvaturas ) * abs( curvaturas )^(1 / 3 )
  } else if ( func == "sigmoid" ) {
    curvaturas <- sigmoid( curvaturas, 0.5 )
  }

  # Quantize training set curvatures
  min_curv <- min( curvaturas, na.rm = TRUE )
  max_curv <- max( curvaturas, na.rm = TRUE )
  disc_curv <- quantize( curvaturas, min_curv, max_curv, k = n_levels )

  # Find k-nearest neighbors for the test set
  knn_results <- FNN::get.knnx( train, query = test, k = k, algorithm = "kd_tree" )

  for ( i in 1:n_test ) {
    distances <- knn_results$distance[ i, ]
    neighs <- knn_results$nn.index[ i, ]

    patch <- rbind( test[ i, , drop = FALSE ], train[ neighs, ] )
    curvature <- point_curvature_estimation( patch )

    if ( func == "log" ) {
      if ( curvature <= 0 ) {
        curvature <- 1e-6
      }
      curvature <- log( curvature )
    } else if ( func == "cubic_root" ) {
      curvature <- sign( curvature ) * abs( curvature )^( 1 / 3 )
    } else if ( func == "sigmoid" ) {
      curvature <- sigmoid( curvature, 0.5 )
    }

    curvaturas_updated <- c( curvaturas, curvature )
    disc_curv_updated <- quantize( curvaturas_updated, min( curvaturas_updated, na.rm = TRUE ), max( curvaturas_updated, na.rm = TRUE ), k = n_levels )
    less <- disc_curv_updated[ length( disc_curv_updated ) ]

    if ( length( neighs ) > 1 && less < length( neighs ) ) {
      neighs <- neighs[ 1:( k - less ) ]
    }

    if ( length( neighs ) %% 2 == 0 ) {
      neighs <- neighs[ 1:( length( neighs ) - 1 ) ]
    }

    # Majority vote
    labels[i] <- as.numeric( names( which.max( table( train_target[ neighs ] ) ) ) )
  }
  return( labels )
}
