% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cbh_metrics.R
\name{get_cbh_metrics}
\alias{get_cbh_metrics}
\title{Methods to estimated the Crown Base Height of a tree: maximum LAD percentage, maximum distance and the last distance}
\usage{
get_cbh_metrics(effective_LAD, min_height= 1.5, hdepth1_height = 2.5, verbose=TRUE)
}
\arguments{
\item{effective_LAD}{Tree metrics with gaps (distances), fuel base heights, and depths of fuel layers with LAD percentage greater than a threshold (10%).
(output of [get_layers_lad()] function).
An object of the class text.}

\item{min_height}{Numeric value for the actual minimum base height (in meters).}

\item{hdepth1_height}{Numeric value for the depth height of the first fuel layer. If the first fuel layer has the maximum LAD and its depth is greater than the indicated value,
then this fuel layer is considered as the CBH of the tree. On the contrary, if its depth is <= the value, the CBH with maximum LAD will be the second fuel layer,
although it has not the maximum LAD.}

\item{verbose}{Logical, indicating whether to display informational messages (default is TRUE).}
}
\value{
A data frame giving the Crown Base Height (CBH) of a tree using three criteria: maximum LAD percentage, maximum distance and the last distance.
For the case of maximum LAD CBH, the output gives the actual CBH with maximum LAD and also, the CBH from the second fuel layer when the first fuel layer has the maximum LAD
but its depth is lesser than the value indicated in the parameter "hdepth1_height".
}
\description{
This function determines the CBH of a segmented tree using three criteria: maximum LAD percentage, maximum distance and the last distance.
}
\details{
List of tree metrics:
\itemize{
  \item treeID: tree ID with strings and numeric values
  \item treeID1: tree ID with only numeric values
  \item dptf: Depth of fuel layers (m) after considering distances greater than the actual height bin step
  \item effdist: Effective distance between consecutive fuel layers (m) after considering distances greater than any number of steps
  \item Hcbh: Base height of each fuel separated by a distance greater than the certain number of steps
  \item Hdptf: Height of the depth of fuel layers (m) after considering distances greater than the actual step
  \item Hdist: Height of the distance (> any number of steps) between consecutive fuel layers (m)
  \item Hcbh_Hdptf - Percentage of LAD values comprised in each effective fuel layer
  \item maxlad_Hcbh - Height of the CBH of the segmented tree based on the maximum LAD percentage
  \item maxlad1_Hcbh - Height of the CBH from the second fuel layer when the maximum LAD occurred in the first fuel layer but its depth <= "hdepth1_height"
  \item max_Hcbh - Height of the CBH of the segmented tree based on the maximum distance found in its profile
  \item last_Hcbh - Height of the CBH of the segmented tree based on the last distance found in its profile
  \item maxlad_ - Values of distance and fuel depth and their corresponding heights at the maximum LAD percentage
  \item maxlad1_ - Values of distance and fuel depth and their corresponding heights for the second fuel layer when the maximum LAD occurred in the first fuel layer but its depth <= "hdepth1_height"
  \item max_ - Values of distance and fuel depth and their corresponding heights at the maximum distance
  \item last_ - Values of distance and fuel depth and their corresponding heights at the last distance
  \item nlayers - Number of effective fuel layers
  \item max_height - Maximum height of the tree profile
}
}
\examples{
library(magrittr)
library(stringr)
library(dplyr)

# Before running this example, make sure to run get_real_depths().
if (interactive()) {
effective_LAD <- get_layers_lad()
LadderFuelsR::effective_LAD$treeID <- factor(LadderFuelsR::effective_LAD$treeID)

trees_name1 <- as.character(effective_LAD$treeID)
trees_name2 <- factor(unique(trees_name1))

cbh_dist_list <- list()

for (i in levels(trees_name2)) {
tree1 <- effective_LAD |> dplyr::filter(treeID == i)
cbh_dist_metrics <- get_cbh_metrics(tree1, min_height= 1.5,  hdepth1_height = 2.5, verbose=TRUE)
cbh_dist_list[[i]] <- cbh_dist_metrics
}

# Combine the individual data frames
cbh_metrics <- dplyr::bind_rows(cbh_dist_list)

# Get original column names
original_column_names <- colnames(cbh_metrics)

# Specify prefixes
desired_order <- c("treeID", "Hcbh", "dptf","effdist","dist", "Hdist", "Hdptf", "max_","last_",
"maxlad_","maxlad1_", "nlayers")

# Identify unique prefixes
prefixes <- unique(sub("^([a-zA-Z]+).*", "\\\\1", original_column_names))
# Initialize vector to store new order
new_order <- c()

# Loop over desired order of prefixes
for (prefix in desired_order) {
 # Find column names matching the current prefix
matching_columns <- grep(paste0("^", prefix), original_column_names, value = TRUE)
# Append to the new order
new_order <- c(new_order, matching_columns)
}
# Reorder values
cbh_metrics <- cbh_metrics[, new_order]
}
}
\seealso{
\code{\link{get_layers_lad}}
}
\author{
Olga Viedma, Carlos Silva, JM Moreno and A.T. Hudak
}
