#' Adjusted P-Values for Benjamini & Bogomolov (2014) BH-q BH-Rq/m Procedure
#'
#' Given a list/data frame of grouped p-values, selecting thresholds and p-value combining method, retruns adjusted p-values to make decisions
#'
#' @usage
#' avgFDR.p.adjust(pval, t, make.decision)
#' @param pval the structural p-values, the type should be \code{"list"}.
#' @param t the thresholds determining whether the families are selected or not, also affects conditional p-value within families.
#' @param make.decision logical; if \code{TRUE}, then the output include the decision rules compared adjusted p-values with significant level \eqn{\alpha}.
#' @return A list of the adjusted conditional p-values, a list of \code{NULL} means the family is not selected to do the test in the second stage.
#' @author Yalin Zhu
#' @references Benjamini, Y., & Bogomolov, M. (2014).
#' Selective inference on multiple families of hypotheses.
#' \emph{Journal of the Royal Statistical Society: Series B (Statistical Methodology)}, \strong{76}: 297-318.
#' @examples
#' # data is from Example 4.1 in Mehrotra and Adewale (2012)
#' pval <- list(c(0.031,0.023,0.029,0.005,0.031,0.000,0.874,0.399,0.293,0.077),
#'              c(0.216,0.843,0.864),
#'              c(1,0.878,0.766,0.598,0.011,0.864),
#'              c(0.889,0.557,0.767,0.009,0.644),
#'              c(1,0.583,0.147,0.789,0.217,1,0.02,0.784,0.579,0.439),
#'              c(0.898,0.619,0.193,0.806,0.611,0.526,0.702,0.196))
#' avgFDR.p.adjust(pval = pval, t=0.1)
#' sum(unlist(avgFDR.p.adjust(pval = pval,t=0.1)) <= 0.1)
#' @export
avgFDR.p.adjust <- function(pval, t, make.decision=FALSE){
  # if (class(pval)!="data.frame" & class(pval)!="list"){
  #   stop("The class of pval has to be 'data.frame' or 'list'.")
  # }
  # if (class(pval)=="data.frame"){
  #   colnames(pval) <- c("family","individual")
  #   pval <- dlply(pval, .(family))
  # }
  adjcondP <- vector("list", length(pval))
  adj.pval <- lapply(pval, FUN = p.adjust, method="BH")
  for (i in which(p.adjust(sapply(adj.pval,min),method="BH")<=t) ){
    adjcondP[[i]] <- p.adjust(pval[[i]], method = "BH")
  }
  if (make.decision==FALSE){return(adjcondP)}
  else if(make.decision==TRUE){
    sig.level=sum(p.adjust(sapply(adj.pval,min),method="BH")<=t)/length(pval)*t
    f <- function(x){ifelse(x<=sig.level,"reject","not reject")}
    return(list(adjust.p = adjcondP,adjust.sig.level=sig.level,dec.make=lapply(adjcondP, f)))
    }
}
