## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
#install.packages("MariNET")
library("MariNET")

## ----load-data----------------------------------------------------------------
# Load the dataset from the package
data(example_data)

# Display the first few rows
head(example_data)


## -----------------------------------------------------------------------------
# Extract column names from the dataset
# These represent all available variables in the dataset
varLabs <- colnames(example_data)

# Define a list of variables to be removed from the analysis
# These variables are not included as nodes in the network visualization
remove <- c("id", "day", "beep", "conc")

# Filter out the unwanted variables
# Keeps only the variables that are not in the "remove" list
varLabs <- varLabs[!varLabs %in% remove]

# Print the final list of selected variables to be used as nodes in the network
print(varLabs)

## -----------------------------------------------------------------------------
# Perform Linear Mixed Model (LMM) analysis on the dataset
# This function iterates over selected variables (varLabs) and models their relationships
# while accounting for individual-level variability using a random effect.

model <- lmm_analysis(
  example_data,   # Input dataset containing clinical/longitudinal data
  varLabs,        # List of selected variables to be analyzed in the model
  random_effects = "(1|id)"  # Specifies a random intercept for each individual (id)
)

# Print the model results (optional, useful for debugging or reviewing output)
# print(model)


## ----qgraph-plot2, fig.width=3, fig.height=2, dpi=300-------------------------
# Define the community structure for the variables
# Assigns labels to different groups based on symptoms or categories
community_structure <- c(
  rep("Stress", 8),   # First 8 variables belong to the "Stress" group
  rep("Social", 6),   # Next 6 variables belong to the "Social" group
  rep("Covid-19", 4)  # Last 4 variables belong to the "Covid-19" group
)

# Create a dataframe linking variable names to their assigned community group
structure <- data.frame(varLabs, community_structure)

# Define labels for the network plot (using variable names)
labels <- varLabs

# Load the qgraph package for network visualization
library(qgraph)

# Generate the network plot using qgraph
qgraph(
  model,                                # Adjacency matrix or network model input
  groups = structure$community_structure, # Assign colors based on community groups
  labels = labels,                        # Display variable names as node labels
  legend = TRUE,                           # Include a legend in the plot
  layout = "spring",                       # Use a force-directed "spring" layout for better visualization
  color = c("orange", "lightblue", "#008080"), # Define colors for different groups
  legend.cex = 0.3                          # Adjust the size of the legend text
)



## ----qgraph-plot3, fig.width=3, fig.height=2, dpi=300-------------------------
# Fit a second Linear Mixed Model (LMM) with a more complex random effects structure
# This model accounts for repeated measures within individuals (id) over different days (day)
# and also considers an additional random effect for the variable "conc" (context or condition)

model2 <- lmm_analysis(
  example_data,    # Input dataset containing clinical/longitudinal data
  varLabs,         # List of selected variables to be analyzed in the model
  random_effects = "(1|id/day) + (1|conc)"  # Random effects structure:
                                            # (1|id/day) -> Nested random effect for each day within an individual
                                            # (1|conc) -> Additional random effect for "conc" variable
)

# Generate a network visualization from the second LMM model
qgraph(
  model2,                                # Adjacency matrix or network model derived from LMM
  groups = structure$community_structure, # Assign colors based on predefined symptom groups
  labels = labels,                        # Display variable names as node labels
  legend = TRUE,                           # Include a legend in the plot
  layout = "spring",                       # Use a force-directed "spring" layout for better visualization
  color = c("orange", "lightblue", "#008080"), # Define colors for different variable groups
  legend.cex = 0.3                          # Adjust the legend text size to avoid oversized labels
)


## ----qgraph-plot4, fig.width=3, fig.height=2, dpi=300-------------------------
# Compute the difference between the two Linear Mixed Model (LMM) networks
# This highlights changes in relationships when considering different random effect structures
difference <- differentiation(model, model2)

# Generate a network visualization of the differences between the two models
qgraph(
  difference,                            # Adjacency matrix representing differences between model1 and model2
  groups = structure$community_structure, # Assign colors based on predefined variable groups
  labels = labels,                        # Display variable names as node labels
  legend = TRUE,                           # Include a legend in the plot
  layout = "spring",                       # Use a force-directed "spring" layout for better visualization
  color = c("orange", "lightblue", "#008080"), # Define colors for different variable groups
  legend.cex = 0.3                          # Adjust legend text size to keep it readable
)


## -----------------------------------------------------------------------------
sessionInfo()

## ----echo=FALSE, results="asis"-----------------------------------------------

