% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/createLocationID.R
\name{createLocationID}
\alias{createLocationID}
\title{Create one or more unique locationIDs}
\usage{
createLocationID(
  longitude = NULL,
  latitude = NULL,
  algorithm = c("geohash", "digest"),
  precision = 10,
  invalidID = as.character(NA)
)
}
\arguments{
\item{longitude}{Vector of longitudes in decimal degrees E.}

\item{latitude}{Vector of latitudes in decimal degrees N.}

\item{algorithm}{Algorithm to use -- either \code{"geohash"} or \code{"digest"}.}

\item{precision}{\code{precision} argument used when encoding with \code{"geohash"}.}

\item{invalidID}{Identifier to use for invalid locations. This can be a
character string or \code{NA}.}
}
\value{
Vector of character locationIDs.
}
\description{
A locationID is created for each incoming \code{longitude} and
\code{latitude}. Each locationID is unique to within a certain spatial scale.
With \code{algorithm = "geohash"}, the
\code{precision} argument determines the size of a geohash grid cell. At the
equator, the following grid cell sizes apply for different precision levels:

\preformatted{
precision   (maximum grid cell X axis, in m)
        5   ± 2400
        6   ± 610
        7   ± 76
        8   ± 19
        9   ± 2.4
       10   ± 0.6
}

Invalid locations will be assigned a locationID specified by the user with
the \code{invalidID} argument, typically \code{NA}.
}
\details{
When the \code{"geohash"} algorithm is specified,
the following code is used to generate each locationID:

\preformatted{
  locationID <-
    geohashTools::gh_encode(latitude, longitude, precision)
}

When the \code{"digest"} algorithm is specified,
the following code is used:

\preformatted{
# Retain accuracy up to ~.1m
locationString <- paste0(
  sprintf("\%.7f", longitude),
  "_",
  sprintf("\%.7f", latitude)
)
# Avoid collisions until billions of records
locationID <- digest::digest(locationString, algo = "xxhash64")
}

See the references for details on either algorithm.
}
\note{
The \code{"geohash"} algorithm is preferred but the \code{"digest"}
algorithm is retained because several existing databases
use the \code{"digest"} algorithm as a unique identifier.
}
\examples{
library(MazamaCoreUtils)

longitude <- c(-122.5, 0, NA, -122.5, -122.5)
latitude <- c( 47.5, 0, 47.5, NA, 47.5)

createLocationID(longitude, latitude)
createLocationID(longitude, latitude, precision = 7)
createLocationID(longitude, latitude, invalidID = "bad")
createLocationID(longitude, latitude, algorithm = "digest")

}
\references{
\url{https://en.wikipedia.org/wiki/Decimal_degrees}

\url{https://www.johndcook.com/blog/2017/01/10/probability-of-secure-hash-collisions/}

\url{https://michaelchirico.github.io/geohashTools/index.html}
}
