# Estimation of effective and basic reproduction numbers (Rt and R0)
# =============================================================================
# EFFECTIVE REPRODUCTION NUMBER (Rt)
# =============================================================================


#' Compute effective reproduction number Rt
#'
#' Estimates the effective reproduction number (Rt) over time using incidence data and
#' a generation time distribution.
#'
#' @param inc Numeric vector of incidence counts.
#' @param G Numeric vector representing the generation time distribution.
#' @return Numeric vector of Rt values.
#' @export
RT <- function(inc, G) {
  m <- length(inc)
  n <- length(G)
  R <- numeric(m)

  for (i in 2:m) {
    t <- 0
    for (j in 1:n) {
      idx <- abs(i - j)
      if (idx > 0 && idx <= length(inc)) {
        t <- t + inc[idx] * G[j]
      }
    }
    R[i] <- ifelse(t > 0, inc[i] / t, 0)
  }

  return(R)
}

# =============================================================================
# BASIC REPRODUCTION NUMBER (R0) - MAXIMUM LIKELIHOOD
# =============================================================================


#' Estimate R0 using maximum likelihood
#'
#' Computes the basic reproduction number (R0) using a maximum likelihood approach
#' based on secondary cases and generation time weights.
#'
#' @param W Generation time distribution.
#' @param N Secondary case counts.
#' @return Estimated R0 value.
#' @export
est_r0_ml <- function(W, N) {
  m <- length(N)
  v <- numeric(m)

  for (i in 2:m) {
    t <- 0
    for (j in 1:min(i, length(W))) {
      idx <- i - j + 1
      if (idx > 0 && idx <= length(N)) {
        t <- t + N[idx] * W[j]
      }
    }
    v[i] <- t
  }

  R0 <- ifelse(sum(v[2:m]) > 0, sum(N[2:m]) / sum(v[2:m]), 0)
  return(R0)
}

# =============================================================================
# BASIC REPRODUCTION NUMBER (R0) - EMPIRICAL ESTIMATION
# =============================================================================


#' Empirical R0 estimation using growth model
#'
#' Estimates the basic reproduction number (R0) using a growth model (exponential or logistic)
#' fitted to incidence data.
#'
#' @param incidence Numeric vector of incidence counts.
#' @param serial_interval Mean serial interval.
#' @param growth_model Type of growth model: "exponential" or "logistic".
#' @return Estimated R0 value.
#' @importFrom stats lm coef
#' @export
empirical_r0 <- function(incidence, serial_interval, growth_model = "exponential") {
  if (length(incidence) < 2) {
    stop("Incidence data must contain at least two points.")
  }
  if (serial_interval <= 0) {
    stop("Serial interval must be greater than zero.")
  }

  incidence <- as.numeric(incidence)
  time_points <- 1:length(incidence)

  tryCatch({
    if (growth_model == "exponential") {
      log_inc <- log(incidence + 1e-10)
      growth_rate <- coef(lm(log_inc ~ time_points))[2]
    } else if (growth_model == "logistic") {
      logit_inc <- log(incidence / (1 + incidence) + 1e-10)
      growth_rate <- coef(lm(logit_inc ~ time_points))[2]
    } else {
      stop(paste("Unsupported growth model:", growth_model))
    }
    return(1 + growth_rate * serial_interval)
  }, error = function(e) {
    stop(paste("Error in empirical R0 estimation:", e$message))
  })
}

# =============================================================================
# BASIC REPRODUCTION NUMBER (R0) - MLE ESTIMATION
# =============================================================================

#' MLE estimation of R0 using generation time
#'
#' Estimates the basic reproduction number (R0) using maximum likelihood and a parametric
#' model based on generation time and incidence data.
#'
#' @param incidence Numeric vector of incidence counts.
#' @param gen_time Mean generation time.
#' @return Estimated R0 value.
#' @importFrom stats optim
#' @export
est_r0_mle <- function(incidence, gen_time) {
  if (length(incidence) < 2) {
    stop("Incidence data must contain at least two points.")
  }
  if (gen_time <= 0) {
    stop("Generation time must be greater than zero.")
  }

  neg_log_likelihood <- function(r0) {
    ll <- 0
    for (t in 2:length(incidence)) {
      time_diffs <- (t - 1):1
      weights <- exp(-time_diffs / gen_time)
      lambda_t <- r0 * sum(incidence[1:(t - 1)] * weights)
      ll <- ll - lambda_t + incidence[t] * log(lambda_t + 1e-10)
    }
    return(-ll)
  }

  tryCatch({
    result <- optim(par = 2.0, fn = neg_log_likelihood,
                    method = "L-BFGS-B",
                    lower = 0.1, upper = 5)
    if (result$convergence != 0) {
      stop(paste("MLE optimization failed:", result$message))
    }
    return(result$par)
  }, error = function(e) {
    stop(paste("Error in MLE R0 estimation:", e$message))
  })
}
