# Clean environment and load libraries
rm(list = ls())  # Clear environment to avoid conflicts


# Source files
library(MixStable)

# ===== TEST FUNCTIONS =====

# Test all ECF functions
test_all_ecf_functions <- function() {
  cat("=== Testing All ECF Functions ===\n")

  # Generate test data
  set.seed(123)
  n <- 500
  test_data <- rt(n, df = 3) * 2 + 1  # Student-t as proxy
  u_grid <- seq(0.1, 1.0, length.out = 20)

  functions_to_test <- list(
    "CDF" = CDF,
    "ecf_regression" = ecf_regression,
    "robust_ecf_regression" = robust_ecf_regression,
    "estimate_stable_recursive_ecf" = estimate_stable_recursive_ecf,
    "estimate_stable_kernel_ecf" = estimate_stable_kernel_ecf,
    "estimate_stable_weighted_ols" = estimate_stable_weighted_ols,
    "estimate_stable_from_cdf" = estimate_stable_from_cdf,
    "ecf_estimate_all" = function(x, u) ecf_estimate_all(x, u)
  )

  results <- list()

  for (func_name in names(functions_to_test)) {
    cat("\nTesting", func_name, "...\n")
    tryCatch({
      if (func_name == "fit_stable_ecf") {
        result <- fit_stable_ecf(test_data, u_grid)
      } else {
        result <- functions_to_test[[func_name]](test_data, u_grid)
      }

      cat("  Alpha:", result$alpha, "\n")
      cat("  Beta:", result$beta, "\n")
      cat("  Gamma:", result$gamma, "\n")
      cat("  Delta:", result$delta, "\n")

      # Check if parameters are finite
      if (all(sapply(result, is.finite))) {
        cat("  ✓ All parameters finite\n")
      } else {
        cat("  ✗ Some parameters not finite\n")
      }

      results[[func_name]] <- result

    }, error = function(e) {
      cat("  ✗ Error:", e$message, "\n")
      results[[func_name]] <- NA
    })
  }

  return(results)
}

# Test Bayesian mixture model
test_bayesian_mixture <- function() {
  cat("\n=== Testing Bayesian Mixture Model ===\n")

  if (!requireNamespace("mixtools", quietly = TRUE)) {
    cat("Skipping Bayesian test - mixtools package not available\n")
    return(NULL)
  }

  # Generate mixture data
  set.seed(456)
  n <- 300
  component <- rbinom(n, 1, 0.4)
  test_data <- ifelse(component == 1,
                      rnorm(n, mean = -1, sd = 1),
                      rnorm(n, mean = 2, sd = 1.5))

  tryCatch({
    mixture_result <- bayesian_mixture_model(test_data, draws = 100, chains = 2)

    cat("✓ Mixture model fitted successfully\n")
    cat("Component weights:", mixture_result$weights, "\n")
    cat("Component means:", mixture_result$means, "\n")
    cat("Component sigmas:", mixture_result$sigmas, "\n")

    return(mixture_result)
  }, error = function(e) {
    cat("✗ Error in mixture model:", e$message, "\n")
    return(NULL)
  })
}

# Run all tests
run_all_tests <- function() {
  cat("=== RUNNING ALL STABLE DISTRIBUTION TESTS ===\n\n")

  # Test ECF functions
  ecf_results <- test_all_ecf_functions()

  # Test Bayesian mixture
  mixture_result <- test_bayesian_mixture()

  # Summary
  cat("\n=== TEST SUMMARY ===\n")
  successful_tests <- sum(!is.na(ecf_results))
  total_tests <- length(ecf_results)

  cat("ECF Functions tested:", total_tests, "\n")
  cat("Successful tests:", successful_tests, "\n")
  cat("Success rate:", round(100 * successful_tests / total_tests, 1), "%\n")

  if (!is.null(mixture_result)) {
    cat("✓ Bayesian mixture model test passed\n")
  } else {
    cat("✗ Bayesian mixture model test failed\n")
  }

  cat("\nAll tests completed!\n")

  return(list(
    ecf_results = ecf_results,
    mixture_result = mixture_result
  ))
}

# To run tests:
results <- run_all_tests()
