#' Generates the standard barplot of scores for each sample
#'
#' This function takes the quality score data generated by accumulate.zscores()
#' and returns the barplot if no filename is specified. If filename is
#' is specified it saves the plot as file and returns NULL. It will also draw a
#' cut-off for which samples to exclude. get.qc.barplot offers a standard template for generating
#' a QC barplot, but can also take any parameter that BoutrosLab.plotting.general::create.barplot
#' takes for more customizability.
#'
#' @param quality.scores A dataframe with columns 'Sum' (of scores) and 'Sample', i.e. the output of accumulate.zscores
#' @param abline.h Adds a horizontal line to the plot; useful for depicting the threshold for what is deemed a poor sample quality score
#' @param yaxis.cex Size of y-axis tick labels, defaults to 0.8
#' @param xaxis.cex Size of x-axis tick labels, defaults to 0
#' @param xaxis.tck	Specifies the length of the tick marks for x-axis, defaults to 0
#' @param xlab.label The label for the x-axis, defaults to ''
#' @param ylab.label label for the y-axis, defaults to 'Sum of Z (Z < 0)'
#' @param ylab.cex Size of y-axis label, defaults to 1
#' @param abline.col Colour of the horizontal line on the plot, defaults to 'darkgrey'
#' @param ... The function can also take any parameter that BoutrosLab.plotting.general::create.barplot takes
#'
#' @inheritParams BoutrosLab.plotting.general::create.barplot
#'
#' @return The barplot or NULL depending if filename is specified
#'
#' @export
get.qc.barplot <- function(
    quality.scores,
    filename = NULL,
    abline.h = -20,
    yaxis.cex = 0.8,
    xaxis.cex = 0,
    yaxis.tck = 1,
    xaxis.tck = 0,
    xlab.label = '',
    ylab.label = 'Sum of Z (Z < 0)',
    ylab.cex = 1,
    abline.col = 'darkgrey',
    axes.lwd = 1,
    ...
    ) {

    # Error checking
    accumulate.zscores.output.check(quality.scores);

    barplot <- BoutrosLab.plotting.general::create.barplot(
        filename = filename,
        Sum ~ Sample,
        quality.scores,
        # Formatting
        yaxis.cex = yaxis.cex,
        xaxis.cex = xaxis.cex,
        yaxis.tck = yaxis.tck,
        xaxis.tck = xaxis.tck,
        xlab.label = xlab.label,
        ylab.label = ylab.label,
        ylab.cex = ylab.cex,
        # Line
        abline.h = abline.h,
        abline.col = abline.col,
        ...
        );
    }
