% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grf.R
\name{get_outcome_weights.causal_forest}
\alias{get_outcome_weights.causal_forest}
\title{Outcome weights for the \code{\link[grf]{causal_forest}} function}
\usage{
\method{get_outcome_weights}{causal_forest}(
  object,
  ...,
  S,
  newdata = NULL,
  S.tau = NULL,
  target = "CATE",
  checks = TRUE
)
}
\arguments{
\item{object}{An object of class \code{causal_forest}, i.e. the result of running \code{\link[grf]{causal_forest}}.}

\item{...}{Pass potentially generic \link{get_outcome_weights} options.}

\item{S}{A smoother matrix reproducing the outcome predictions used in building the \code{\link[grf]{instrumental_forest}}.
Obtained by calling \code{get_forest_weights()} for the \code{\link[grf]{regression_forest}} object producing the outcome predictions.}

\item{newdata}{Corresponds to \code{newdata} option in \code{\link[grf]{predict.causal_forest}}. If \code{NULL},
out-of-bag outcome weights, otherwise for those for the provided test data returned.}

\item{S.tau}{Required if \code{target != "CATE"}, then S.tau is the CATE smoother obtained from running \code{get_outcome_weights()}
with \code{target == "CATE"}.}

\item{target}{Target parameter for which outcome weights should be extracted. Currently \code{c("CATE","ATE")} implemented.}

\item{checks}{Default \code{TRUE} checks whether weights numerically replicate original estimates. Only set \code{FALSE} if you
know what you are doing and need to save computation time.}
}
\value{
\link{get_outcome_weights} object with \code{omega} containing weights and \code{treat} the treatment
}
\description{
Post-estimation command to extract outcome weights for causal forest
implemented via the \code{\link[grf]{causal_forest}} function from the \pkg{grf} package.
}
\examples{
\donttest{
# Sample from DGP borrowed from grf documentation
n = 500
p = 10
X = matrix(rnorm(n * p), n, p)
W = rbinom(n, 1, 0.5)
Y = pmax(X[, 1], 0) * W + X[, 2] + pmin(X[, 3], 0) + rnorm(n)

# Run outcome regression and extract smoother matrix
forest.Y = grf::regression_forest(X, Y)
Y.hat = predict(forest.Y)$predictions
outcome_smoother = grf::get_forest_weights(forest.Y)

# Run causal forest with external Y.hats
c.forest = grf::causal_forest(X, Y, W, Y.hat = Y.hat)

# Predict on out-of-bag training samples.
cate.oob = predict(c.forest)$predictions

# Predict using the forest.
X.test = matrix(0, 101, p)
X.test[, 1] = seq(-2, 2, length.out = 101)
cate.test = predict(c.forest, X.test)$predictions

# Calculate outcome weights
omega_oob = get_outcome_weights(c.forest,S = outcome_smoother)
omega_test = get_outcome_weights(c.forest,S = outcome_smoother,newdata = X.test)

# Observe that they perfectly replicate the original CATEs
all.equal(as.numeric(omega_oob$omega \%*\% Y), 
          as.numeric(cate.oob))
all.equal(as.numeric(omega_test$omega \%*\% Y), 
          as.numeric(cate.test))

# Also the ATE estimates are perfectly replicated
omega_ate = get_outcome_weights(c.forest,target = "ATE", 
                                S = outcome_smoother, 
                                S.tau = omega_oob$omega)
all.equal(as.numeric(omega_ate$omega \%*\% Y),
          as.numeric(grf::average_treatment_effect(c.forest, target.sample = "all")[1]))

# The omega weights can be plugged into balancing packages like cobalt
}

}
\references{
Athey, S., Tibshirani, J., & Wager, S. (2019). Generalized random forest. The Annals of Statistics, 47(2), 1148-1178.

Knaus, M. C. (2024). Treatment effect estimators as weighted outcomes, \url{https://arxiv.org/abs/2411.11559}.
}
