#' @title Print Methods for Various Objects
#' 
#' @description
#' Print concise, user-friendly summaries of objects generated by the Qval package.
#' Supports objects of classes \code{\link[Qval]{CDM}}, \code{\link[Qval]{validation}}, \code{\link[Qval]{sim.data}}, 
#' \code{\link[Qval]{fit}}, \code{\link[Qval]{is.Qident}}, \code{\link[Qval]{att.hierarchy}}, 
#' as well as their corresponding summary objects.
#'
#' @details
#' The \code{print} methods provide an at-a-glance view of key information:
#' \describe{
#'   \item{\code{print.CDM}}{displays sample size, item and attribute counts, and package information.}
#'   \item{\code{print.validation}}{shows suggested modifications to the Q-matrix, marking changed entries with an asterisk.}
#'   \item{\code{print.sim.data}}{reports dimensions of simulated data and offers guidance on extraction.}
#'   \item{\code{print.fit}}{show basic fit indices.}
#'   \item{\code{print.is.Qident}}{prints basic results from \code{is.Qident}.}
#'   \item{\code{print.att.hierarchy}}{prints basic results from \code{att.hierarchy}.}
#'   \item{\code{print.summary.CDM}}{prints fitted model details and alpha-pattern distribution from a \code{summary.CDM} object.}
#'   \item{\code{print.summary.validation}}{prints suggested Q-matrix changes or a message if none are recommended.}
#'   \item{\code{print.summary.sim.data}}{prints attribute-pattern frequencies and proportions from \code{summary.sim.data}.}
#'   \item{\code{print.summary.fit}}{prints basic fit indices from \code{summary.fit}.}
#'   \item{\code{print.summary.is.Qident}}{prints basic results from \code{summary.is.Qident}.}
#'   \item{\code{print.summary.att.hierarchy}}{prints basic results from \code{summary.att.hierarchy}.}
#' }
#'
#' @param x An object of the appropriate class (e.g., \code{\link[Qval]{CDM}}, 
#'          \code{\link[Qval]{validation}}, \code{\link[Qval]{sim.data}}, 
#'          \code{\link[Qval]{fit}}, \code{\link[Qval]{is.Qident}}, 
#'          \code{\link[Qval]{att.hierarchy}}, or their summaries).
#' @param ... Currently unused.  Additional arguments are ignored.
#' 
#' @return Invisibly returns \code{x}.
#'
#' @examples
#' set.seed(123)
#' library(Qval)
#' 
#' \donttest{
#' ################################################################
#' # Example 1: print a CDM object                                #
#' ################################################################
#' Q <- sim.Q(3, 20)
#' IQ <- list(P0 = runif(20, 0.0, 0.2), P1 = runif(20, 0.8, 1.0))
#' data.obj <- sim.data(Q = Q, N = 500, IQ = IQ, 
#'                      model = "GDINA", distribute = "horder")
#' CDM.obj <- CDM(data.obj$dat, Q, model = "GDINA", 
#'                method = "EM", maxitr = 2000, verbose = 1)
#' print(CDM.obj)
#' 
#'
#' ################################################################
#' # Example 2: print a validation object                         #
#' ################################################################
#' set.seed(123)
#' MQ <- sim.MQ(Q, 0.1)
#' CDM.obj <- CDM(data.obj$dat, MQ)
#' validation.obj <- validation(data.obj$dat, MQ, CDM.obj, 
#'                              method = "GDI")
#' print(validation.obj)
#' 
#'
#' ################################################################
#' # Example 3: print a sim.data object                           #
#' ################################################################
#' set.seed(123)
#' Q2 <- sim.Q(3, 10)
#' data.obj2 <- sim.data(Q = Q2, N = 1000)
#' print(data.obj2)
#' 
#' 
#' ################################################################
#' # Example 4: print a fit object                           #
#' ################################################################
#' set.seed(123)
#' Q2 <- sim.Q(3, 10)
#' fit.obj <- fit(Y = data.obj$dat, Q = Q, model = "GDINA")
#' print(fit.obj)
#' 
#' 
#' ################################################################
#' # Example 5: print summary objects                             #
#' ################################################################
#' summary.CDM.obj <- summary(CDM.obj)
#' print(summary.CDM.obj)
#' 
#' summary.val.obj <- summary(validation.obj)
#' print(summary.val.obj)
#' 
#' summary.sim.obj <- summary(data.obj2)
#' print(summary.sim.obj)
#' 
#' summary.fit <- summary(fit.obj)
#' print(summary.fit)
#' }
#'
#' @name print
NULL

#' @describeIn print Print method for \code{CDM} objects
#' @export
print.CDM <- function(x, ...) {
  printPackageInfo()
  out <- summary(x)
  cat("==============================================\n")
  cat(" Number of items       =", out$base[2], "\n",
      "Number of attributes  =", out$base[3], "\n",
      "Number of individuals =", out$base[1], "\n",
      "To extract components, use the method extract.\n")
  invisible(x)
}

#' @describeIn print Print method for \code{validation} objects
#' @export
print.validation <- function(x, ...) {
  printPackageInfo()
  cat("==============================================\n")
  Q.sug <- data.frame(extract.validation(x, "Q.sug"))
  Q.orig <- data.frame(extract.validation(x, "Q.orig"))
  if (any(Q.sug != Q.orig)) {
    cat("\nSuggested Q-matrix: \n\n")
    Q.sug[Q.sug != Q.orig] <- paste0(Q.sug[Q.sug != Q.orig], "*")
    print(Q.sug, right = FALSE)
    cat("Note: * denotes a modified element.\n")
  } else {
    cat("\nNo Q-matrix modifications are suggested.\n")
  }
  invisible(x)
}

#' @describeIn print Print method for \code{sim.data} objects
#' @export
print.sim.data <- function(x, ...) {
  printPackageInfo()
  cat("==============================================\n")
  cat(" Number of items       =", nrow(x$Q), "\n",
      "Number of attributes  =", ncol(x$Q), "\n",
      "Number of individuals =", nrow(x$dat), "\n",
      "To extract components, use the method extract.\n")
  invisible(x)
}

#' @describeIn print Print method for \code{fit} objects
#' @export
print.fit <- function(x, ...) {
  printPackageInfo()
  npar <- x$npar
  refit <- c(x$`-2LL`, x$AIC, x$BIC, x$CAIC, x$SABIC)
  names(refit) <- c("-2LL", "AIC", "BIC", "CAIC", "SABIC")
  abfit <- c(x$M2, x$SRMSR)
  names(abfit) <- c("M2", "df", "p.value", "RMSEA2", "SRMSR")

  cat("==============================================\n")
  cat(" Number of parameters  =", npar, "\n\n")
  cat("Relative fit indices  = :\n")
  print(round(refit, 3))
  cat("\nAbsolute fit indices = :\n") 
  print(round(abfit, 3))
  invisible(x)
}

#' @describeIn print Print method for \code{is.Qident} objects
#' @export
print.is.Qident <- function(x, ...) {
  printPackageInfo()
  cat("==============================\nUnder DINA or DINO model:\n", 
      "    Completeness:", x$strictIdentifiability.obj$completeness, "\n", 
      "    Distinctness:", x$strictIdentifiability.obj$distinctness, "\n", 
      "    Repetition:  ", x$strictIdentifiability.obj$repetition, "\n", 
      " therefore, the Q-matrix", 
      ifelse((x$strictIdentifiability.obj$completeness & 
                x$strictIdentifiability.obj$distinctness & 
                x$strictIdentifiability.obj$repetition), 
             "is", "is not"), 
      "Joint Strict Identifiability.\n")
  
  cat("==============================\nUnder DINA or DINO model:\n", 
      "    Locally Generic Completeness:", x$genericIdentifiability.DINA.obj$locallyGenericIdentifiability, "\n", 
      "    Globally Generic Repetition: ", x$genericIdentifiability.DINA.obj$globallyGenericIdentifiability, "\n", 
      " therefore, the Q-matrix is", 
      ifelse(x$genericIdentifiability.DINA.obj$locallyGenericIdentifiability, 
             ifelse(x$genericIdentifiability.DINA.obj$globallyGenericIdentifiability, "Globally", "Locally"), 
             "not Joint"), 
      "Generic Identifiability.\n")
  
  cat("==============================\nUnder saturated model:\n", 
      "    Generic Completeness:", x$genericIdentifiability.obj$genericCompleteness, "\n", 
      "    Generic Repetition:  ", x$genericIdentifiability.obj$genericRepetition, "\n", 
      " therefore, the Q-matrix", 
      ifelse((x$genericIdentifiability.obj$genericCompleteness & 
                x$genericIdentifiability.obj$genericRepetition), 
             "is", "is not"), 
      "Joint Generic Identifiability.\n")
  
  invisible(x)
}

#' @describeIn print Print method for \code{att.hierarchy} objects
#' @export
print.att.hierarchy <- function(x, ...) {
  printPackageInfo()
  
  cat("==============================\n", 
      paste0("All > ", x$arguments$eps, " and significantly > 0 (alpha=0.01) structural parameters:\n\n"))
  print(data.frame(matrix(sprintf("%.4f", x$statistic),
                          nrow = nrow(x$statistic),
                          dimnames = dimnames(x$statistic))))
  cat("\n")
  cat("The convergence", ifelse(x$isNonverge, "was", "was not"), "achieved.")
  if(x$noSig){
    warning("Warning: none of the structural parameters are significantly greater than 0, causing the procedure to be terminated!")
  }
  
  invisible(x)
}

#' @describeIn print Print method for \code{summary.CDM} objects
#' @export
print.summary.CDM <- function(x, ...) {
  cat("==============================================\n")
  cat(" Number of items       =", x$base[2], "\n",
      "Number of attributes  =", x$base[3], "\n",
      "Number of individuals =", x$base[1], "\n")
  cat("\nModel Fit:\n")
  print(x$model.fit)
  cat("\nDistribution of Alpha Patterns:\n")
  print(x$patterns)
  invisible(x)
}

#' @describeIn print Print method for \code{summary.validation} objects
#' @export
print.summary.validation <- function(x, ...) {
  cat("==============================================\n")
  Q.sug <- x$Q.sug
  Q.orig <- x$Q.orig
  if (any(Q.sug != Q.orig)) {
    cat("\nSuggested Q-matrix: \n\n")
    Q.sug[Q.sug != Q.orig] <- paste0(Q.sug[Q.sug != Q.orig], "*")
    print(Q.sug, right = FALSE)
    cat("Note: * denotes a modified element.\n")
  } else {
    cat("\nNo Q-matrix modifications are suggested.\n")
  }
  invisible(x)
}

#' @describeIn print Print method for \code{summary.sim.data} objects
#' @export
print.summary.sim.data <- function(x, ...) {
  cat("==============================================\n")
  cat(" Number of items       =", x$base[2], "\n",
      "Number of attributes  =", x$base[3], "\n",
      "Number of individuals =", x$base[1], "\n")
  
  if(!is.null(x$att.str)){
    distribute <- paste0(sapply(x$att.str, 
                                function(x) return(paste0(paste0("A", x[1]), "->", paste0("A", x[2])))), 
                         collapse = ", ")
    wrapped.distribute <- strwrap(distribute, width = 80)
    cat("attribute structure: \n", paste(wrapped.distribute, collapse = "\n"), "\n")
  }
  
  cat("\nDistribution of Alpha Patterns:\n")
  print(x$patterns)
  invisible(x)
}

#' @describeIn print Print method for \code{summary.fit} objects
#' @export
print.summary.fit <- function(x, ...) {
  npar <- x$npar
  refit.print <- x$fit.relative
  abfit.print <- x$fit.absolute
  
  cat("==============================================\n")
  cat(" Number of parameters  =", npar, "\n\n")
  cat("Relative fit indices  = :\n")
  print(round(refit.print, 3))
  cat("\nAbsolute fit indices = :\n") 
  print(round(abfit.print, 3))
  invisible(x)
}

#' @describeIn print Print method for \code{summary.is.Qident} objects
#' @export
print.summary.is.Qident <- function(x, ...) {
  cat("==============================\nUnder DINA or DINO model:\n", 
      "    Completeness:", x$strictIdentifiability.obj$completeness, "\n", 
      "    Distinctness:", x$strictIdentifiability.obj$distinctness, "\n", 
      "    Repetition:  ", x$strictIdentifiability.obj$repetition, "\n", 
      " therefore, the Q-matrix", 
      ifelse((x$strictIdentifiability.obj$completeness & 
                x$strictIdentifiability.obj$distinctness & 
                x$strictIdentifiability.obj$repetition), 
             "is", "is not"), 
      "Joint Strict Identifiability.\n")
  
  cat("==============================\nUnder DINA or DINO model:\n", 
      "    Locally Generic Completeness:", x$genericIdentifiability.DINA.obj$locallyGenericIdentifiability, "\n", 
      "    Globally Generic Repetition: ", x$genericIdentifiability.DINA.obj$globallyGenericIdentifiability, "\n", 
      " therefore, the Q-matrix is", 
      ifelse(x$genericIdentifiability.DINA.obj$locallyGenericIdentifiability, 
             ifelse(x$genericIdentifiability.DINA.obj$globallyGenericIdentifiability, "Globally", "Locally"), 
             "not Joint"), 
      "Generic Identifiability.\n")
  
  cat("==============================\nUnder saturated model:\n", 
      "    Generic Completeness:", x$genericIdentifiability.obj$genericCompleteness, "\n", 
      "    Generic Repetition:  ", x$genericIdentifiability.obj$genericRepetition, "\n", 
      " therefore, the Q-matrix", 
      ifelse((x$genericIdentifiability.obj$genericCompleteness & 
                x$genericIdentifiability.obj$genericRepetition), 
             "is", "is not"), 
      "Joint Generic Identifiability.\n")
  invisible(x)
}

#' @describeIn print Print method for \code{summary.att.hierarchy} objects
#' @export
print.summary.att.hierarchy <- function(x, ...) {
  cat("==============================\n", 
      paste0("All > ", x$eps, " and significantly > 0 (alpha=0.01) structural parameters:\n\n"))
  print(data.frame(matrix(sprintf("%.4f", x$statistic),
                          nrow = nrow(x$statistic),
                          dimnames = dimnames(x$statistic))))
  cat("\n")
  cat("The convergence", ifelse(x$isNonverge, "was", "was not"), "achieved.")
  if(x$noSig){
    warning("Warning: none of the structural parameters are significantly greater than 0, causing the procedure to be terminated!")
  }
  
  invisible(x)
}
