#' SIFT output filter
#'
#' This function takes as input the output file generated by the SIFT-MS
#' and returns a .csv containing the TIME and the concentrations data selected
#' by the user
#'
#' @param setdir allow the selection of the working directory
#' @param input_name allow the selection of the input file
#' @param output_name name of the .csv output file
#' @param n_parameters number of analytes
#' @param param_names vector with name of the analytes
#' @param out_file flag for the export of a csv file
#'
#' @return Filtered data and optional csv from SIFT input
#'
#' @examples
#' data(raw_SIFT)
#' SIFT_output_filter(input_name = raw_SIFT, output_name = "testfile", out_file = FALSE)
#'
#'
#' @importFrom utils read.csv read.delim write.csv
#'
#' @export




SIFT_output_filter <- function(setdir = getwd(), input_name = file.choose(),
                               output_name, n_parameters = 2, param_names = c("Isoprene", "Acetone"),
                               out_file = TRUE){
  oldwd <- getwd()
  on.exit(setwd(oldwd))
  setwd(setdir)

  if (typeof(input_name) == "character") {
    dat = read.delim(input_name)
  } else {
    dat = input_name
  }


  for (i in 1:nrow(dat)) {
    if ("Analyte vs Time,:" %in% dat[i,1]) {
      begin = i
    }
    if ("Summary,:" %in% dat[i,1]) {
      end = i
    }
  }

  dat <- t(dat[(begin+2):(end-1),1])

  new_dat <- matrix(nrow = ncol(dat), ncol = n_parameters+2)

  for (i in 1:nrow(new_dat)) {
    holder <- strsplit(dat[1,i], ",")

    for (j in 1:ncol(new_dat)) {
      if (j != 2) {
        new_dat[i,j] <- as.numeric(holder[[1]][j])
      }
    }
  }

  new_dat <- data.frame(new_dat)
  new_dat <- new_dat[-2]
  names(new_dat) <- append(c("Time"),param_names)
  new_dat$Time <- mapply('/', as.numeric(new_dat$Time), 60000)

  if (out_file == TRUE) {
    write.csv(new_dat,file = paste(output_name, ".csv", sep = ""), row.names = F)
  }
  return(new_dat)

}


