#' @title Airfield game
#' @description Given an airfield problem, this function returns the associated airfield game.
#' @param c A vector of costs defining the airfield problem.
#' @param binary A logical value. By default, \code{binary=FALSE}.
#' @return The characteristic function of the airfield game, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details Let \eqn{N = \{1, \dots, n\}} denote a set of agents, and let \eqn{c \in \mathbb{R}_+^N} be a cost vector.
#' Each \eqn{c_i} represents the cost of the service required by agent \eqn{i}.
#' Segmental costs are defined as the difference between a given cost and the first immediately lower cost: \eqn{c_i - c_{i-1}} for \eqn{i \in N \backslash \{1\}}.
#'
#' Each \eqn{c \in \mathbb{R}_+^N} defines an airfield problem, which is associated to an airfield game \eqn{v_{a}\in G^N}, is defined by
#' \deqn{v_{a}(S)=\max\{c_j:j\in S\}\text{ for all }S\in 2^N.}
#'
#' Airfield games, as defined, are cost games, but they can also be expressed as savings games.
#' Additional tools and methods for addressing airfield problems are available
#' in the \pkg{AirportProblems} package \cite{Bernárdez Ferradás et al. (2025)}.
#' @seealso \link{claimsgame}, \link{savingsgame}
#' @examples
#' c <- c(2000,3200,4100,5100)
#' airfieldgame(c,binary=TRUE)
#' @references Bernárdez Ferradás, A., Sánchez Rodríguez, E., Mirás Calvo, M., & Quinteiro Sandomingo, C. (2025). \emph{AirportProblems: Analysis of Cost Allocation for Airport Problems}. R package version 0.1.0. \url{https://CRAN.R-project.org/package=AirportProblems}
#' @references Littlechild, S.C., & Owen, G. (1973). A Simple Expression for the Shapely Value in a Special Case. \emph{Management Science}, 23, 370-372.
#' @export

airfieldgame <- function(c, binary = FALSE) {

  ################################
  ###Comprobación datos entrada###
  ################################

  if (sum(c<=0)!=0)
  {
    stop("'c' must be positive.")
  }

  ################################
  #####Creación de variables######
  ################################

  n=length(c)#Número de jugadores
  nC=(2^n)-1#Número de coaliciones.
  v=c() #Airfield game

  ################################
  ######Cuerpo de la función######
  ################################

  for(S in 1:((nC-1)/2))#Por dualidad, recorremos solo la mitad de las coaliciones.
  {
    #Calculamos los jugadores que pertenecen a la coalición S.
    num=S
    J=floor(log2(S))+1
    Sjug=c()

    for (ii in 1:J)
    {
      if (num %% 2 ==1)
      {
        Sjug=c(Sjug,ii)
      }
      num=num %/% 2
    }
    #Valor del juego del aeropuerto:
    v[S]=max(c[Sjug])
    #Aplicamos dualidad:
    NmenosSjug=c(1:n)[-Sjug]
    v[nC-S]=max(c[NmenosSjug])
  }

  v[nC]=max(c)

  ################################
  ######Salidas de la función#####
  ################################

  if (binary == FALSE) { # Devolvemos el juego en orden lexicográfico.
    return(bin2lex(v))
  } else { # Devolvemos el juego en orden binario.
    return(v)
  }
}# Fin de la función
