#' @title Kohlberg criterion for the prenucleolus
#' @description This function applies the Kohlberg criterion to check if the given efficient allocation is the prenucleolus of the given game.
#' @param v A characteristic function, as a vector.
#' @param x An efficient allocation, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return \code{TRUE} if \code{x} is the prenucleolus of \code{v}, \code{FALSE} otherwise.
#' @details Given \eqn{v \in G^{N}} and \eqn{x \in \mathbb{R}^{n}} with \eqn{\sum_{i \in N} x_{i} = v(N)},
#' let \eqn{k(x)} be the number of different excesses in \eqn{x}.
#' According to the Kohlberg criterion for the prenucleolus, \eqn{x} is the prenucleolus of \eqn{v}
#' if and only if, for each \eqn{j \in \{1,\dots,k(x)\}}, \eqn{\bigcup_{t=1}^{j} F^{t}}
#' is a balanced family, being \eqn{F^{t}} the set of coalitions associated with the excess
#' that occupies position \eqn{t} when excesses are arranged in decreasing order.
#' @examples
#' v <- c(0,0,0,0,10,40,30,60,10,20,90,90,90,130,160)
#' x <- prenucleolusvalue(v)
#' kohlbergcriterion(v, x) # x is the prenucleolus of v
#' y <- prenucleolusvalue(v) + c(1,-1,0,0)
#' kohlbergcriterion(v, y) # y is not the prenucleolus of v
#'
#' # If the game is 0-monotonic, its nucleolus coincides with its prenucleolus,
#' # and therefore must pass the Kohlberg criterion for the prenucleolus:
#' v4 <- c(-2,-2,-2,7,7,7,6)
#' zeromonotoniccheck(v4)
#' kohlbergcriterion(v4, nucleolusvalue(v4))
#' @seealso \link{balancedfamilycheck}, \link{excesses}, \link{prenucleolusvalue}
#' @references Kohlberg, E. (1971). On the Nucleolus of a Characteristic Function Game. \emph{SIAM Journal on Applied Mathematics}, 20(1), 62–66.
#' @export

kohlbergcriterion <- function(v, x, binary = FALSE, tol = 100*.Machine$double.eps) {
  # kohlbergcheck Checks if a preimputation is the prenucleolus of a game

  # % INPUT
  # v   = Characteristic function in binary order
  # x = a preimputation
  # tol = tolerance (by default 100*eps)
  # OUTPUT
  # check=1 if x is the prenucleolus of v.
  # check=0 otherwise

  # Datos
  nC <- length(v)            # Número de coaliciones
  n <- log2(nC + 1)          # Número de jugadores

  # Comprobamos que x es una preimputación
  if (length(x) != n || abs(sum(x) - v[nC]) > tol) {
    stop("'x' is not an efficient allocation for 'v'.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  check <- TRUE

  ###############
  # Excess vector
  ###############
  xS <- numeric(nC - 1)

  # Construcción de los incrementos: x1, x2 - x1, x3 - (x1 + x2), ...
  acumula <- cumsum(x)
  # Primera coalicion en el orden invertido
  incrementos <- c(x[1], x[2:n] - acumula[1:(n - 1)])

  xS[1] <- x[1]
  for (c in 1:(nC - 2)) {
    # % (no incluimos la coalicion S=N con exceso nulo siempre)
    # Ojo, ahora escribo en binario de derecha a izquierda
    # Busco la posición m del primer cero (desde la posición más a la derecha)
    # en la "nueva" representacion binaria de la coalicion c.
    # Es decir, el primer jugador que no pertenece a c
    # Primer jugador no en S
    m <- 0
    ii <- 1
    cc <- c
    while (m == 0) {
      if (cc %% 2 == 0) {
        m <- ii
      }
      ii <- ii + 1
      cc <- floor(cc / 2)
    }
    xS[c + 1] <- xS[c] + incrementos[m] # pago de cada coalicion
  }

  # Vector de excesos ordenados de mayor a menor
  # y coaliciones donde se alcanzan esos valores
  E <- v[1:(nC - 1)] - xS
  ord <- order(-E)  # Orden descendente
  E <- E[ord]
  Coal <- ord
  ########################
  # Criterio de Kohlberg
  ########################
  pos <- 1 # Posicion de la última coalicion utilizada
  Fam <- Coal[1]
  while (check == TRUE && pos < nC - 2) {
    beta <- which(abs(E - E[pos]) < tol) # Coaliciones en las que se alcanza un nuevo exceso
    Fam <- unique(c(Fam, Coal[beta])) # Fam es la unión de la familia anterior con las coaliciones en las que se alcanza el nuevo exceso
    if (balancedfamilycheck(Fam, n, tol)$check == FALSE) { #Si Fam no es equilibrada
      check <- FALSE  # x no es el prenucleolo
    }
    pos <- beta[length(beta)] + 1
  }

  return(check)
}
