% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mvrandn.R
\name{mvrandn}
\alias{mvrandn}
\title{Truncated multivariate normal generator}
\usage{
mvrandn(l, u, Sig, n, mu = NULL)
}
\arguments{
\item{l}{lower truncation limit}

\item{u}{upper truncation limit}

\item{Sig}{covariance matrix}

\item{n}{number of simulated vectors}

\item{mu}{location parameter}
}
\value{
a \eqn{d} by \eqn{n} matrix storing the random vectors, \eqn{X}, drawn from \eqn{N(0,\Sigma)}, conditional on \eqn{l<X<u};
}
\description{
Simulate \eqn{n}  independent and identically distributed random vectors
 from the \eqn{d}-dimensional \eqn{N(0,\Sigma)} distribution
 (zero-mean normal with covariance \eqn{\Sigma}) conditional on \eqn{l<X<u}.
Infinite values for \eqn{l} and \eqn{u} are accepted.
}
\details{
\itemize{
\item Bivariate normal:
Suppose we wish to simulate a bivariate \eqn{X} from \eqn{N(\mu,\Sigma)}, conditional on
\eqn{X_1-X_2<-6}. We can recast this as the problem of simulation
of \eqn{Y} from \eqn{N(0,A\Sigma A^\top)} (for an appropriate matrix \eqn{A})
conditional on \eqn{l-A\mu < Y < u-A\mu} and then setting \eqn{X=\mu+A^{-1}Y}.
   See the example code below.
 \item Exact posterior simulation for Probit regression: Consider the
     Bayesian Probit Regression model applied to the \code{\link{lupus}} dataset.
     Let the prior for the regression coefficients \eqn{\beta} be \eqn{N(0,\nu^2 I)}. Then, to simulate from the Bayesian
     posterior exactly, we first simulate
     \eqn{Z} from \eqn{N(0,\Sigma)}, where  \eqn{\Sigma=I+\nu^2 X X^\top,}
     conditional on \eqn{Z\ge 0}. Then, we simulate the posterior regression coefficients, \eqn{\beta}, of the Probit regression
     by drawing \eqn{(\beta|Z)} from \eqn{N(C X^\top Z,C)}, where \eqn{C^{-1}=I/\nu^2+X^\top X}.
See the example code below.
}
}
\note{
The algorithm may not work or be very inefficient if \eqn{\Sigma} is close to being rank deficient.
}
\examples{
 # Bivariate example.

 Sig <- matrix(c(1,0.9,0.9,1), 2, 2);
 mu <- c(-3,0); l <- c(-Inf,-Inf); u <- c(-6,Inf);
 A <- matrix(c(1,0,-1,1),2,2);
 n <- 1e3; # number of sampled vectors
 Y <- mvrandn(l - A \%*\% mu, u - A \%*\% mu, A \%*\% Sig \%*\% t(A), n);
 X <- rep(mu, n) + solve(A, diag(2)) \%*\% Y;
 # now apply the inverse map as explained above
 plot(X[1,], X[2,]) # provide a scatterplot of exactly simulated points
\dontrun{
# Exact Bayesian Posterior Simulation Example.

data("lupus"); # load lupus data
Y = lupus[,1]; # response data
X = lupus[,-1]  # construct design matrix
m=dim(X)[1]; d=dim(X)[2]; # dimensions of problem
 X=diag(2*Y-1) \%*\%X; # incorporate response into design matrix
 nu=sqrt(10000); # prior scale parameter
 C=solve(diag(d)/nu^2+t(X)\%*\%X);
 L=t(chol(t(C))); # lower Cholesky decomposition
 Sig=diag(m)+nu^2*X \%*\% t(X); # this is covariance of Z given beta
 l=rep(0,m);u=rep(Inf,m);
 est=mvNcdf(l,u,Sig,1e3);
 # estimate acceptance probability of Crude Monte Carlo
 print(est$upbnd/est$prob)
 # estimate the reciprocal of acceptance probability
 n=1e4 # number of iid variables
 z=mvrandn(l,u,Sig,n);
 # sample exactly from auxiliary distribution
 beta=L \%*\% matrix(rnorm(d*n),d,n)+C \%*\% t(X) \%*\% z;
 # simulate beta given Z and plot boxplots of marginals
 boxplot(t(beta))
 # plot the boxplots of the marginal
 # distribution of the coefficients in beta
 print(rowMeans(beta)) # output the posterior means
 }
}
\seealso{
\code{\link{mvNqmc}}, \code{\link{mvNcdf}}
}
\keyword{internal}
