% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/UNCOVER.R
\name{plot.UNCOVER}
\alias{plot.UNCOVER}
\title{Plot various outputs of UNCOVER}
\usage{
\method{plot}{UNCOVER}(
  x,
  type = "covariates",
  plot_var = x$Minimum_Spanning_Tree_Variables,
  diagnostic_x_axis = "full",
  ...
)
}
\arguments{
\item{x}{Object of class \code{"UNCOVER"}}

\item{type}{Can be one of; \code{"covariates"} for cluster assignment
visualisation for the co-variates, \code{"fitted"} for co-variate visualisation
with respect to their fitted values, \code{"samples"} for posterior visualisation
or \code{"diagnostics"} for diagnostic plots. See details. Defaults to
\code{"covariates"}.}

\item{plot_var}{Vector specifying which columns (or associated logistic
regression coefficients) of the co-variate matrix should be plotted. Does not
apply when \code{type=="diagnostics"}. Defaults to all columns being selected.}

\item{diagnostic_x_axis}{Only applies if \verb{"type=="diagnostics"}. Either
\code{"full"} (default) for all observations indices to be plotted on the x-axis
or \code{"minimal"} for only some observations indices to be plotted on the
x-axis.}

\item{...}{Arguments to be passed to methods}
}
\value{
No return value, called for side effects
}
\description{
Allows visualisation of many aspects of UNCOVER, including
co-variate, posterior and diagnostic plots.
}
\details{
If \code{type=="covariates"}, the resulting plot will be a ggpairs plot.
The diagonal entries will be a plot of K density plots (K being the number
of clusters). The off-diagonal elements are scatter-plots of the
observations, given a label according to their true response and a colour
based on their assigned cluster. If \code{length(plot_var)==1} then the density
plot and the scatter-plot are combined. If a cluster contains less than two
data points the density will not be plotted.

If \verb{"type=="fitted"}, the resulting plot will be a ggpairs plot. The
diagonal entries will be two density plots, one for data predicted to have
response 0 by the model (red) and one for training data predicted to have
response 1 by the model (green). The off-diagonal elements are
scatter-plots of the observations, given a label according to their actual
response and a colour scale based on their predicted response. If
\code{length(plot_var)==1} then the density plot and the scatter-plot are
combined. If a predicted class (0 or 1) contains less than two data points
the density will not be plotted.

If \code{type=="samples"}, the resulting plot will be a ggpairs plot of the
clusters posteriors, giving the coefficient densities in the diagonal and
scatter-plots of the posterior samples in the off-diagonal. The transparency
is increased in the upper triangle for scenarios when posteriors overlap.

If \code{"type==diagnostics"}, the resulting plot depends on the deforestation
criterion used to create the \code{"UNCOVER"} object:
\describe{
\item{\code{"None"}}{A plot tracking the overall log Bayesian evidence every time
an action is executed.}
\item{\code{"NoC"}}{A plot tracking the overall log Bayesian evidence after every
action and a plot tracking the number of clusters after every action.}
\item{\code{"SoC"}}{Three plots; one tracking the overall log Bayesian evidence
after every action, one tracking the number of criterion breaking clusters
after every action and one tracking the minimum cluster size after every
action.}
\item{\code{"MaxReg"}}{A plot tracking the overall log Bayesian evidence every
time an action is executed. Actions are coloured and each action has an
associated coloured dashed line indicating the log Bayesian evidence plus
the logarithm of the maximum tolerance provided.}
\item{\code{"Validation"}}{A plot tracking the overall log Bayesian evidence
after every action (for both the training data and all of the data) and a
plot tracking the robustness statistic after every deforestation action.}
\item{\code{"Diverse"}}{Three plots; one tracking the overall log Bayesian
evidence after every action, one tracking the number of criterion breaking
clusters after every action and one tracking the minimum minority class
across clusters after every action.}
}
Actions are defined as either edge removals, edge additions or edge
additions in the deforestation stage. The syntax for an action will be the
'type_of_action.edge'. For example the removal of an edge connecting
observation 1 and observation 2 will be displayed 'Rem.1-2'. If the edge was
being added this would be displayed 'Def.Add.1-2' if in the deforestation
stage and 'Add.1-2' otherwise. When the data for the \code{"UNCOVER"} object
created is large setting \code{diagnostic_x_axis=="minimal"} is recommended as it
gives a more visually appealing output.
}
\examples{

\donttest{
require(graphics)
# First we generate a co-variate matrix and binary response vector
CM <- matrix(rnorm(200),100,2)
rv <- sample(0:1,100,replace=TRUE)

# We can then run our algorithm for each of the different deforestation
# criteria
UN.none <- UNCOVER(X = CM,y = rv, deforest_criterion = "None", verbose = FALSE)
UN.noc <- UNCOVER(X = CM,y = rv, deforest_criterion = "NoC",
                  options = UNCOVER.opts(max_K = 3), verbose = FALSE)
UN.soc <- UNCOVER(X = CM,y = rv, deforest_criterion = "SoC",
                  options = UNCOVER.opts(min_size = 10), verbose = FALSE)
UN.maxreg <- UNCOVER(X = CM,y = rv, deforest_criterion = "MaxReg",
                     options = UNCOVER.opts(reg = 1), verbose = FALSE)
UN.validation <- UNCOVER(X = CM,y = rv, deforest_criterion = "Validation",
                         options = UNCOVER.opts(train_frac = 0.8),
                         verbose = FALSE)
UN.diverse <- UNCOVER(X = CM,y = rv, deforest_criterion = "Diverse",
                       options = UNCOVER.opts(n_min_class = 2), verbose = FALSE)
plot(UN.none,type = "covariates")
plot(UN.none,type = "fitted")
plot(UN.none,type = "samples")
plot(UN.none,type = "diagnostics",diagnostic_x_axis = "minimal")
plot(UN.noc,type = "diagnostics",diagnostic_x_axis = "minimal")
plot(UN.soc,type = "diagnostics",diagnostic_x_axis = "minimal")
plot(UN.maxreg,type = "diagnostics",diagnostic_x_axis = "minimal")
plot(UN.validation,type = "diagnostics",diagnostic_x_axis = "minimal")
plot(UN.diverse,type = "diagnostics",diagnostic_x_axis = "minimal")

# If we only wanted to view the second co-variate
plot(UN.none,type = "covariates",plot_var=2)
plot(UN.none,type = "fitted",plot_var=2)
plot(UN.none,type = "samples",plot_var=2)
}

}
\seealso{
\code{\link[=UNCOVER]{UNCOVER()}}
}
\keyword{UNCOVER}
\keyword{plot}
