/******************************************************************************
 * @section DESCRIPTION
 *
 * VIC Physical Constants Header File
 *
 * @section LICENSE
 *
 * The Variable Infiltration Capacity (VIC) macroscale hydrological model
 * Copyright (C) 2016 The Computational Hydrology Group, Department of Civil
 * and Environmental Engineering, University of Washington.
 *
 * The VIC model is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *****************************************************************************/

#ifndef VIC_PHYSICAL_CONSTANTS
#define VIC_PHYSICAL_CONSTANTS

/***** Time Conversions *****/
#define DAYS_PER_360DAY_YEAR 360  /**< days in 360day year */
#define DAYS_PER_YEAR 365  /**< days in nonleap year */
#define DAYS_PER_LYEAR 366  /**< days in leap year */
#define DAYS_PER_JYEAR 365.25  /** days in Julian year */
#define HOURS_PER_DAY 24  /**< hours per day */
#define MONTHS_PER_YEAR 12  /**< months per year */
#define MIN_PER_HOUR 60  /**< minutes per hour */
#define MIN_PER_DAY (MIN_PER_HOUR * HOURS_PER_DAY)  /**< hours per day */
#define SEC_PER_MIN 60  /**< seconds per minutes */
#define SEC_PER_HOUR (SEC_PER_MIN * MIN_PER_HOUR)  /**< seconds per hour */
#define SEC_PER_DAY (SEC_PER_HOUR * HOURS_PER_DAY)  /**< hours per day */

/***** Unit Conversions *****/
#define JOULES_PER_CAL 4.1868  /**< Joules per calorie */
#define GRAMS_PER_KG 1000  /**< grams per kilogram */
#define PA_PER_KPA 1000  /**< Pa per kPa */
#define BAR_PER_KPA 100  /**< bars per kPa */
#define RAD_PER_DEG 0.0174532925  /**< radians per degree */
#define M_PER_KM 1000 /**< meters per kilometer */
#define MM_PER_M 1000  /**< milimeters per meter */
#define CM_PER_M 100  /**< centimeters per meter */
#define MM_PER_CM 10  /**< milimeters per centimeter */
#define MM_PER_IN 25.4  /**< milimeters per inch */
#define IN_PER_M (MM_PER_M / MM_PER_IN)  /**< inches per meter */
#define MOLE_PER_KMOLE 1000 /**< moles per kilomole */
#define FRACT_TO_PERCENT 100
#define PPM_to_MIXRATIO 1.0e-6
#define C_TO_F(t) (t * 9. / 5.) + 32.

/***** Mathematical Constants *****/
#define CONST_PI 3.14159265358979323846

/***** Time Constants *****/
#define CONST_CDAY 86400  /**< seconds in calendar day ~ sec */
#define CONST_SDAY 86164  /**< seconds in siderial day ~ sec */
#define CONST_DDAYS_PER_YEAR 365.2425  /**< decimal days in year */

/***** Oribital Constants *****/
#define CONST_OMEGA (2.0 * CONST_PI / CONST_SDAY)
#define CONST_SECPERRAD 13750.9871  /**< seconds per radian of hour angle */
#define CONST_RADPERDAY 0.017214  /**< radians of Earth orbit per julian day */
#define CONST_RADPERDEG 0.01745329  /**< radians per degree */
#define CONST_MINDECL -0.4092797  /**< minimum declination (radians) */
#define CONST_DAYSOFF 11.25  /**< julian day offset of winter solstice */

/***** Physical Constants *****/
#define CONST_REARTH 6.37122e6  /**< radius of the earth ~ m */
#define CONST_G 9.80616  /**< (m s-2) standard gravitational accel. */
#define CONST_STEBOL 5.67e-8  /**< Stefan-Boltzmann constant ~ W/m^2/K^4 */
#define CONST_BOLTZ 1.38065e-23  /**< Boltzmann's constant ~ J/K/molecule */
#define CONST_AVOGAD 6.02214e26  /**< Avogadro's number ~ molecules/kmole */
#define CONST_KARMAN 0.4  /**< Von Karman constant */
/**< molecular weights */
#define CONST_MWDAIR 28.966  /**< molecular weight of dry air ~ kg/kmole */
#define CONST_MWWV 18.016  /**< molecular weight of water vapor ~ kg/kmole */
#define CONST_MWCO2 44.011 /**< molecular weight of CO2 ~ kg/kmole */
#define CONST_MWAIR 28.97  /**< molecular weight of air ~ kg/kmole */
#define CONST_MWC 12.01  /**< molecular weight of carbon ~ kg/kmole */
/**< gas constants */
#define CONST_RGAS (CONST_AVOGAD * CONST_BOLTZ) /**< Universal gas constant ~ J/K/kmole */
#define CONST_RDAIR (CONST_RGAS / CONST_MWDAIR)  /**< Dry air gas constant ~ J/K/kg */
#define CONST_RWV (CONST_RGAS / CONST_MWWV)  /**< Water vapor gas constant ~ J/K/kg */
#define CONST_EPS (CONST_MWWV / CONST_MWAIR)  /**< Ratio of molecular weights */
/**< temperatures */
#define CONST_TKTRIP 273.16  /**< triple point of fresh water ~ K */
#define CONST_TKFRZ 273.15  /**< freezing T of fresh water ~ K */
/**< standard temperature and pressure */
#define CONST_PSTD 101325.0  /**< (Pa) standard pressure at 0.0 m elevation */
#define CONST_TSTD (CONST_TKFRZ + 15.0)  /**< (K) standard temp at 0.0 m elevation */
/**< densities */
#define CONST_RHODAIR (CONST_PSTD / (CONST_RDAIR * CONST_TKFRZ))  /**< density of dry air at STP ~ kg/m^3 */
#define CONST_RHOFW 1.000e3  /**< density of fresh water ~ kg/m^3 */
#define CONST_RHOICE 0.917e3  /**< density of ice   ~ kg/m^3 */
/**< specific heats */
#define CONST_CPDAIR 1.00464e3  /**< (J kg-1 K-1) specific heat of air */
#define CONST_CPMAIR 1.013e3  /**< (J kg-1 K-1) specific heat of moist air */
#define CONST_CPWV 1.810e3  /**< specific heat of water vap ~ J/kg/K */
#define CONST_CPFW 4.188e3  /**< specific heat of fresh h2o ~ J/kg/K */
#define CONST_CPFWICE 4.2e3  /**< specific heat of fresh h2o ~ J/kg/K */
#define CONST_CPICE 2.11727e3  /**< specific heat of fresh ice ~ J/kg/K */
/**< volumetric heats */
#define CONST_VCPICE_WQ (CONST_CPICE * CONST_RHOFW)  /**< heat capacity of fresh ice per volume of water equivalent ~ J/m^3/K */
/**< latent heats */
#define CONST_LATICE 3.337e5  /**< latent heat of fusion ~ J/kg */
#define CONST_LATVAP 2.501e6  /**< latent heat of evaporation ~ J/kg */
#define CONST_LATSUB (CONST_LATICE + CONST_LATVAP)  /**< latent heat of sublimation ~ J/kg */

/**< special values */
#define CONST_SPVAL 1.0e30

#endif
