#################################################
# Functions for solving for entropy weights.
# Originally from: wtsolveSanjaydynamic_updatedcutoff.R
# Dependencies: corpcor package
#################################################

#' @title Finds the weights for entropy estimation 
#' @description Builds the matrix of gamma-ratios per Berrett et al. (2019), eqn (14).
#' @noRd
gmrat <- function(terms, l) {
  # 'terms' is a vector of j-values
  # 'l' is a vector of dimension-based constraints
  rmat <- c()
  for (j in seq_along(l)) {
    rat <- gamma(1 + l[j])                       # first column
    for (i in seq_along(terms)) {
      if (i == 1) {
        if (terms[1] > 1) {
          sp  <- seq(2, terms[1], 1)
          rat <- prod((sp - 1 + l[j]) / (sp - 1)) * rat
        }
      } else {
        sq  <- seq(terms[i - 1] + 1, terms[i], 1)
        rat <- c(rat,
                 prod((sq - 1 + l[j]) / (sq - 1)) * rat[i - 1])
      }
    }
    rmat <- cbind(rmat, rat)
  }
  rmat
}

#' @title Pseudoinverse projection helper
#' @noRd
withPinv <- function(G) {
  M     <- t(G) %*% G
  invM  <- corpcor::pseudoinverse(M)   # or pracma::pinv(M), MASS::ginv(M), etc.
  P     <- G %*% invM %*% t(G)
  oneVec <- rep(1, nrow(G))
  w <- oneVec - P %*% oneVec

  return(as.numeric(w))
}


#' @title Main weight solver
#' @param terms vector of j indices
#' @param d data dimension
#' @param k integer k  (k = n^a, already rounded/floored)
#' @param a exponent 'a' so that k = n^a
#' @param option 1 = plain pseudoinverse, 3 = eigen-rank truncation
#' @noRd
wtsolve <- function(terms, d, k, a, option = 1) {
  # Build 'l' per eqn(14) in Berrett et al.
  l <- if (floor(d / 4) == 0) 1 else seq(1, floor(d / 4), 1)
  l <- 2 * l / d

  n <- length(terms)
  if (n == 1) return(1)

  a=log(n)/log(k)

  ## Construct G
  G <- gmrat(terms, l)

  ## If no constraints, return uniform weights
  if (length(l) == 0) {
    return(rep(1 / n, n))
  }

  # Option 3: transform G with an eigen-based rank approach
  #           then revert to option 1.
  if (option == 3) {
    eigH  <- eigen(t(G) %*% G) # , symmetric = TRUE
    cumprop <- cumsum(eigH$values) / sum(eigH$values)
    rnk     <- max(1, length(which(cumprop <= 0.99999)))

    ## Project G onto the top rnk eigenspace
    G <- G %*% Re(
      t( sqrt(eigH$values[1:rnk]) * t(eigH$vectors[, 1:rnk]) )
    )
    return(withPinv(G))
  }

  ## Default (option 1): straightforward pseudoinverse
  return(withPinv(G))
}
