% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_weather.R
\name{format_weather}
\alias{format_weather}
\title{Format weather data into an an object suitable for use in ascotraceR spore dispersal models}
\usage{
format_weather(
  x,
  YYYY = NULL,
  MM = NULL,
  DD = NULL,
  hh = NULL,
  mm = NULL,
  POSIXct_time = NULL,
  time_zone = NULL,
  temp,
  rain,
  ws,
  wd,
  wd_sd,
  station,
  lon = NULL,
  lat = NULL,
  r = NULL,
  lonlat_file = NULL
)
}
\arguments{
\item{x}{A \link{data.frame} object of weather station data for formatting.
\code{character}.}

\item{YYYY}{Column name or index in \code{x} that refers to the year when the
weather was logged. \code{character}.}

\item{MM}{Column name or index in \code{x} that refers to the month (numerical)
when the weather was logged. \code{character}.}

\item{DD}{Column name or index in \code{x} that refers to the day of month when
the weather was logged. \code{character}.}

\item{hh}{Column name or index in \code{x} that refers to the hour (24 hour) when
the weather was logged. \code{character}.}

\item{mm}{Column name or index in \code{x} that refers to the minute when the
weather was logged. \code{character}.}

\item{POSIXct_time}{Column name or index in \code{x} which contains a \code{POSIXct}
formatted time. This can be used instead of arguments \code{YYYY}, \code{MM}, \code{DD},
\code{hh}, \code{mm.} \code{character}.}

\item{time_zone}{Time zone (Olsen time zone format) where the weather station
is located. May be in a column or supplied as a character string. Optional,
see also \code{r}. \code{character}. See details.}

\item{temp}{Column name or index in \code{x} that refers to temperature in degrees
Celsius. \code{character}.}

\item{rain}{Column name or index in \code{x} that refers to rainfall in mm.
\code{character}.}

\item{ws}{Column name or index in \code{x} that refers to wind speed in km / h.
\code{character}.}

\item{wd}{Column name or index in \code{x} that refers to wind direction in
degrees. \code{character}.}

\item{wd_sd}{Column name or index in \code{x} that refers to wind speed columns
standard deviation .\code{character}.  This is only applicable if weather data
is already summarised to hourly increments. See details.}

\item{station}{Column name or index in \code{x} that refers to the weather station
name or identifier. \code{character}. See details.}

\item{lon}{Column name or index in \code{x} that refers to weather station's
longitude. \code{character}. See details.}

\item{lat}{Column name or index in \code{x} that refers to weather station's
latitude. \code{character}. See details.}

\item{r}{Spatial raster which is intended to be used with this weather data
for use in the blackspot model. Used to set \code{time_zone} if it is not
supplied in data. \code{character}. Optional, see also \code{time_zone}.}

\item{lonlat_file}{A file path to a \acronym{CSV} which included station
name/id and longitude and latitude coordinates if they are not supplied in
the data. \code{character}. Optional, see also \code{lon} and \code{lat}.}
}
\value{
A \code{ascotraceR.weather} object (an extension of \CRANpkg{data.table})
containing the supplied weather aggregated to each hour in a suitable
format for use with \code{\link[=trace_asco]{trace_asco()}} containing the following columns:
\tabular{rl}{
\strong{times}: \tab Time in POSIXct format \cr
\strong{rain}: \tab Rainfall in mm \cr
\strong{ws}: \tab Wind speed in km / h \cr
\strong{wd}: \tab Wind direction in compass degrees \cr
\strong{wd_sd}: \tab Wind direction standard deviation in compass degrees \cr
\strong{lon}: \tab Station longitude in decimal degrees \cr
\strong{lat}: \tab Station latitude in decimal degrees \cr
\strong{station}: \tab Unique station identifying name \cr
\strong{YYYY}: \tab Year \cr
\strong{MM}: \tab Month \cr
\strong{DD}: \tab Day \cr
\strong{hh}: \tab Hour \cr
\strong{mm}: \tab Minute \cr}
}
\description{
Formats raw weather data into an object suitable for use in the
\code{\link[=trace_asco]{trace_asco()}} function ensuring that the supplied weather data meet the
requirements of the model to run.
}
\details{
\code{time_zone} All weather stations must fall within the same time
zone.  If the required stations are located in differing time zones,
separate \code{ascotraceR.weather} objects must be created for each time zone.
If a raster object, \code{r}, of previous crops is provided that spans time
zones, an error will be emitted.

\code{wd_sd} If weather data is provided in hourly increments, a column
with the standard deviation of the wind direction over the hour is required
to be provided. If the weather data are sub-hourly, the standard deviation
will be calculated and returned automatically.

\code{lon}, \code{lat} and \code{lonlat_file} If \code{x} provides longitude and
latitude values for station locations, these may be specified in the \code{lon}
and \code{lat} columns.  If the coordinates are not relevant to the study
location \code{NA} can be specified and the function will drop these column
variables.  If these data are not included, (\code{NULL}) a separate file may be
provided that contains the longitude, latitude and matching station name to
provide station locations in the final \code{ascotraceR.weather} object that is
created by specifying the file path to a \acronym{CSV} file using
\code{lonlat_file}.
}
\examples{
# Weather data files for Newmarracara for testing and examples have been
# included in ascotraceR. The weather data files both are of the same format,
# so they will be combined for formatting here.

Newmarracarra <- read.csv(
 system.file("extdata",
             "1998_Newmarracarra_weather_table.csv",
             package = "ascotraceR")
)

station_data <- system.file("extdata",
                            "stat_dat.csv",
                            package = "ascotraceR")

weather <- format_weather(
  x = Newmarracarra,
  POSIXct_time = "Local.Time",
  temp = "mean_daily_temp",
  rain = "rain_mm",
  ws = "ws",
  wd = "wd",
  wd_sd = "wd_sd",
  station = "Location",
  time_zone = "Australia/Perth",
  lonlat_file = station_data
)

# Saving weather data and reimporting can lose the object class
# Reimported data can be quickly reformatted, adding the 'asco.weather' class
#  with this same function
temp_file_path <- paste0(tempdir(),"weather_file.csv")
write.csv(weather, file = temp_file_path, row.names = FALSE)
weather_imported <- read.csv(temp_file_path)
weather <- format_weather(weather_imported,
                          time_zone = "Australia/Perth")
unlink(temp_file_path) # remove temporary weather file
}
